<?php
/**
 * Age Calculator Widget for Elementor
 * 
 * @package Relayout_Design
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Make sure Elementor is active
if (!class_exists('\Elementor\Widget_Base')) {
    return;
}

class Age_Calculator_Widget extends \Elementor\Widget_Base {

    public function get_name() {
        return 'age_calculator';
    }

    public function get_title() {
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            return __('Age Calculator', 'relayout-design') . ' <span style="color: #ff6b35; font-weight: bold; font-size: 10px; background: #fff3f0; padding: 2px 6px; border-radius: 3px; margin-left: 5px;">PRO</span>';
        }
        return __('Age Calculator', 'relayout-design');
    }

    public function get_icon() {
        return 'eicon-calendar';
    }

    public function get_categories() {
        return ['relayout-designs'];
    }

    public function get_keywords() {
        return ['age', 'calculator', 'date', 'birth', 'years', 'months', 'days'];
    }

    public function get_script_depends() {
        return ['age-calculator-widget-script'];
    }

    public function get_style_depends() {
        return ['age-calculator-widget-style'];
    }

    protected function register_controls() {
        
        // Content Section
        $this->start_controls_section(
            'content_section',
            [
                'label' => __('Content', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'widget_title',
            [
                'label' => __('Widget Title', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Age Calculator', 'relayout-design'),
                'placeholder' => __('Enter widget title...', 'relayout-design'),
            ]
        );

        $this->add_control(
            'date_label',
            [
                'label' => __('Date Input Label', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Enter your date of birth:', 'relayout-design'),
                'placeholder' => __('Enter label text...', 'relayout-design'),
            ]
        );

        $this->add_control(
            'calculate_button_text',
            [
                'label' => __('Calculate Button Text', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Calculate Age', 'relayout-design'),
                'placeholder' => __('Enter button text...', 'relayout-design'),
                'condition' => [
                    'auto_calculate!' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'display_format',
            [
                'label' => __('Display Format', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'full',
                'options' => [
                    'years_only' => __('Years Only', 'relayout-design'),
                    'years_months' => __('Years + Months', 'relayout-design'),
                    'full' => __('Full Details (Years, Months, Days)', 'relayout-design'),
                    'total_days' => __('Total Days', 'relayout-design'),
                    'total_months' => __('Total Months', 'relayout-design'),
                ],
            ]
        );

        $this->add_control(
            'auto_calculate',
            [
                'label' => __('Auto Calculate', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
                'description' => __('Calculate age automatically when date is selected', 'relayout-design'),
            ]
        );

        $this->add_control(
            'show_next_birthday',
            [
                'label' => __('Show Next Birthday', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
                'description' => __('Display days until next birthday', 'relayout-design'),
            ]
        );

        $this->add_control(
            'show_birth_day',
            [
                'label' => __('Show Birth Day', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
                'description' => __('Display the day of the week you were born', 'relayout-design'),
            ]
        );

        $this->end_controls_section();

        // Labels Section
        $this->start_controls_section(
            'labels_section',
            [
                'label' => __('Custom Labels', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'years_label',
            [
                'label' => __('Years Label', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Years', 'relayout-design'),
            ]
        );

        $this->add_control(
            'months_label',
            [
                'label' => __('Months Label', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Months', 'relayout-design'),
            ]
        );

        $this->add_control(
            'days_label',
            [
                'label' => __('Days Label', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Days', 'relayout-design'),
            ]
        );

        $this->add_control(
            'next_birthday_label',
            [
                'label' => __('Next Birthday Label', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Days until next birthday:', 'relayout-design'),
                'condition' => [
                    'show_next_birthday' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'birth_day_label',
            [
                'label' => __('Birth Day Label', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('You were born on a:', 'relayout-design'),
                'condition' => [
                    'show_birth_day' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - General
        $this->start_controls_section(
            'style_general_section',
            [
                'label' => __('General Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'widget_background',
            [
                'label' => __('Background Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-widget' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'widget_border',
                'selector' => '{{WRAPPER}} .age-calculator-widget',
            ]
        );

        $this->add_responsive_control(
            'widget_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 16,
                    'right' => 16,
                    'bottom' => 16,
                    'left' => 16,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-widget' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'widget_shadow',
                'selector' => '{{WRAPPER}} .age-calculator-widget',
            ]
        );

        $this->add_responsive_control(
            'widget_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 30,
                    'right' => 30,
                    'bottom' => 30,
                    'left' => 30,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-widget' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Title
        $this->start_controls_section(
            'style_title_section',
            [
                'label' => __('Title Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .age-calculator-title',
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => __('Title Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'title_margin',
            [
                'label' => __('Margin', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Input
        $this->start_controls_section(
            'style_input_section',
            [
                'label' => __('Input Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'input_typography',
                'selector' => '{{WRAPPER}} .age-calculator-input, {{WRAPPER}} .age-calculator-label',
            ]
        );

        $this->add_control(
            'input_background',
            [
                'label' => __('Input Background', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#f8f9fa',
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-input' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'input_color',
            [
                'label' => __('Input Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-input' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'input_border',
                'selector' => '{{WRAPPER}} .age-calculator-input',
            ]
        );

        $this->add_responsive_control(
            'input_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-input' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'input_padding',
            [
                'label' => __('Input Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'default' => [
                    'top' => 14,
                    'right' => 16,
                    'bottom' => 14,
                    'left' => 16,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-input' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Button
        $this->start_controls_section(
            'style_button_section',
            [
                'label' => __('Button Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'button_typography',
                'selector' => '{{WRAPPER}} .age-calculator-button',
            ]
        );

        $this->start_controls_tabs('button_style_tabs');

        $this->start_controls_tab(
            'button_normal_tab',
            [
                'label' => __('Normal', 'relayout-design'),
            ]
        );

        $this->add_control(
            'button_background',
            [
                'label' => __('Background Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#007cba',
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-button' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-button' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_hover_tab',
            [
                'label' => __('Hover', 'relayout-design'),
            ]
        );

        $this->add_control(
            'button_hover_background',
            [
                'label' => __('Background Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#005a87',
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-button:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_hover_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-button:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'button_border',
                'selector' => '{{WRAPPER}} .age-calculator-button',
            ]
        );

        $this->add_responsive_control(
            'button_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'button_padding',
            [
                'label' => __('Button Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'default' => [
                    'top' => 14,
                    'right' => 24,
                    'bottom' => 14,
                    'left' => 24,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'button_width',
            [
                'label' => __('Button Width', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 100,
                        'max' => 500,
                    ],
                    '%' => [
                        'min' => 10,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .age-calculator-button' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Result
        $this->start_controls_section(
            'style_result_section',
            [
                'label' => __('Result Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'result_typography',
                'selector' => '{{WRAPPER}} .age-result',
            ]
        );

        $this->add_control(
            'result_background',
            [
                'label' => __('Background Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#f0f8ff',
                'selectors' => [
                    '{{WRAPPER}} .age-result' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'result_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}} .age-result' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'result_border',
                'selector' => '{{WRAPPER}} .age-result',
            ]
        );

        $this->add_responsive_control(
            'result_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .age-result' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'result_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .age-result' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        // Check if premium features are available
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            $upgrade_prompt = relayout_designs_show_upgrade_prompt('Age Calculator Widget');
            if ($upgrade_prompt) {
                echo $upgrade_prompt;
                return;
            }
        }
        
        $settings = $this->get_settings_for_display();
        
        $widget_settings = [
            'displayFormat' => $settings['display_format'],
            'autoCalculate' => $settings['auto_calculate'] === 'yes',
            'showNextBirthday' => $settings['show_next_birthday'] === 'yes',
            'showBirthDay' => $settings['show_birth_day'] === 'yes',
            'labels' => [
                'years' => $settings['years_label'],
                'months' => $settings['months_label'],
                'days' => $settings['days_label'],
                'nextBirthday' => $settings['next_birthday_label'],
                'birthDay' => $settings['birth_day_label'],
            ],
        ];
        ?>
        
        <div class="age-calculator-widget" data-settings="<?php echo esc_attr(json_encode($widget_settings)); ?>">
            <?php if (!empty($settings['widget_title'])): ?>
                <h3 class="age-calculator-title"><?php echo esc_html($settings['widget_title']); ?></h3>
            <?php endif; ?>
            
            <div class="age-calculator-form">
                <?php if (!empty($settings['date_label'])): ?>
                    <label class="age-calculator-label" for="birth-date"><?php echo esc_html($settings['date_label']); ?></label>
                <?php endif; ?>
                
                <div class="input-group">
                    <input type="date" id="birth-date-<?php echo esc_attr($this->get_id()); ?>" class="age-calculator-input" max="<?php echo esc_attr(date('Y-m-d')); ?>" aria-label="<?php echo esc_attr($settings['date_label']); ?>">
                    <?php if ($settings['auto_calculate'] !== 'yes'): ?>
                        <button type="button" class="age-calculator-button" aria-label="<?php echo esc_attr($settings['calculate_button_text']); ?>">
                            <?php echo esc_html($settings['calculate_button_text']); ?>
                        </button>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="age-result" style="display: none;" role="region" aria-live="polite" aria-label="Age calculation result">
                <div class="age-display"></div>
                
                <?php if ($settings['show_next_birthday'] === 'yes'): ?>
                    <div class="next-birthday-info" style="display: none;"></div>
                <?php endif; ?>
                
                <?php if ($settings['show_birth_day'] === 'yes'): ?>
                    <div class="birth-day-info" style="display: none;"></div>
                <?php endif; ?>
            </div>
        </div>
        
        <?php
    }

    protected function content_template() {
        ?>
        <#
        var widgetId = 'birth-date-' + view.getID();
        var widgetSettings = {
            displayFormat: settings.display_format || 'full',
            autoCalculate: settings.auto_calculate === 'yes',
            showNextBirthday: settings.show_next_birthday === 'yes',
            showBirthDay: settings.show_birth_day === 'yes',
            labels: {
                years: settings.years_label || 'Years',
                months: settings.months_label || 'Months',
                days: settings.days_label || 'Days',
                nextBirthday: settings.next_birthday_label || 'Days until next birthday:',
                birthDay: settings.birth_day_label || 'You were born on a:'
            }
        };
        #>
        
        <div class="age-calculator-widget" data-settings="{{ JSON.stringify(widgetSettings) }}">
            <# if (settings.widget_title) { #>
                <h3 class="age-calculator-title">{{{ settings.widget_title }}}</h3>
            <# } #>
            
            <div class="age-calculator-form">
                <# if (settings.date_label) { #>
                    <label class="age-calculator-label" for="{{ widgetId }}">{{{ settings.date_label }}}</label>
                <# } #>
                
                <div class="input-group">
                    <input type="date" id="{{ widgetId }}" class="age-calculator-input" max="{{ new Date().toISOString().split('T')[0] }}" aria-label="{{{ settings.date_label }}}">
                    <# if (settings.auto_calculate !== 'yes') { #>
                        <button type="button" class="age-calculator-button" aria-label="{{{ settings.calculate_button_text }}}">
                            {{{ settings.calculate_button_text || 'Calculate Age' }}}
                        </button>
                    <# } #>
                </div>
            </div>
            
            <div class="age-result" style="display: none;" role="region" aria-live="polite" aria-label="Age calculation result">
                <div class="age-display">
                    <div class="age-container">
                        <div class="age-item">
                            <span class="age-number">25</span>
                            <span class="age-label">{{{ settings.years_label || 'Years' }}}</span>
                        </div>
                        <# if (settings.display_format === 'years_months' || settings.display_format === 'full') { #>
                            <div class="age-item">
                                <span class="age-number">6</span>
                                <span class="age-label">{{{ settings.months_label || 'Months' }}}</span>
                            </div>
                        <# } #>
                        <# if (settings.display_format === 'full') { #>
                            <div class="age-item">
                                <span class="age-number">15</span>
                                <span class="age-label">{{{ settings.days_label || 'Days' }}}</span>
                            </div>
                        <# } #>
                    </div>
                </div>
                
                <# if (settings.show_next_birthday === 'yes') { #>
                    <div class="next-birthday-info">
                        <div class="next-birthday">{{{ settings.next_birthday_label || 'Days until next birthday:' }}} <strong>120 days</strong></div>
                    </div>
                <# } #>
                
                <# if (settings.show_birth_day === 'yes') { #>
                    <div class="birth-day-info">
                        <div class="birth-day">{{{ settings.birth_day_label || 'You were born on a:' }}} <strong>Monday</strong></div>
                    </div>
                <# } #>
            </div>
        </div>
        <?php
    }
}
