/**
 * Age Calculator Widget JavaScript
 * Handles age calculation and display functionality
 */

class AgeCalculatorWidget {
    constructor(element) {
        this.element = element;
        this.settings = JSON.parse(element.dataset.settings || '{}');
        this.dateInput = element.querySelector('.age-calculator-input');
        this.calculateButton = element.querySelector('.age-calculator-button');
        this.resultContainer = element.querySelector('.age-result');
        this.ageDisplay = element.querySelector('.age-display');
        this.nextBirthdayInfo = element.querySelector('.next-birthday-info');
        this.birthDayInfo = element.querySelector('.birth-day-info');
        
        this.init();
    }

    init() {
        this.bindEvents();
        
        // Set max date to today
        const today = new Date().toISOString().split('T')[0];
        this.dateInput.setAttribute('max', today);
    }

    bindEvents() {
        if (this.settings.autoCalculate) {
            this.dateInput.addEventListener('change', () => this.calculateAge());
            this.dateInput.addEventListener('input', () => this.calculateAge());
        } else if (this.calculateButton) {
            this.calculateButton.addEventListener('click', () => this.calculateAge());
        }
    }

    calculateAge() {
        const birthDate = new Date(this.dateInput.value);
        
        if (!this.dateInput.value || isNaN(birthDate.getTime())) {
            this.hideResult();
            return;
        }

        const today = new Date();
        
        // Validate birth date is not in the future
        if (birthDate > today) {
            this.showError('Birth date cannot be in the future!');
            return;
        }

        const age = this.calculateDetailedAge(birthDate, today);
        this.displayAge(age, birthDate);
        this.showResult();
    }

    calculateDetailedAge(birthDate, currentDate) {
        let years = currentDate.getFullYear() - birthDate.getFullYear();
        let months = currentDate.getMonth() - birthDate.getMonth();
        let days = currentDate.getDate() - birthDate.getDate();

        // Adjust for negative days
        if (days < 0) {
            months--;
            const lastMonth = new Date(currentDate.getFullYear(), currentDate.getMonth(), 0);
            days += lastMonth.getDate();
        }

        // Adjust for negative months
        if (months < 0) {
            years--;
            months += 12;
        }

        // Calculate total days
        const totalDays = Math.floor((currentDate - birthDate) / (1000 * 60 * 60 * 24));
        
        // Calculate total months (approximate)
        const totalMonths = years * 12 + months;

        // Calculate next birthday
        const nextBirthday = new Date(currentDate.getFullYear(), birthDate.getMonth(), birthDate.getDate());
        if (nextBirthday < currentDate) {
            nextBirthday.setFullYear(currentDate.getFullYear() + 1);
        }
        const daysUntilBirthday = Math.ceil((nextBirthday - currentDate) / (1000 * 60 * 60 * 24));

        // Get birth day of week
        const birthDayOfWeek = this.getDayOfWeek(birthDate);

        return {
            years,
            months,
            days,
            totalDays,
            totalMonths,
            daysUntilBirthday,
            birthDayOfWeek
        };
    }

    displayAge(age, birthDate) {
        let ageText = '';
        
        switch (this.settings.displayFormat) {
            case 'years_only':
                ageText = `<div class="age-item">
                    <span class="age-number">${age.years}</span>
                    <span class="age-label">${this.settings.labels.years}</span>
                </div>`;
                break;
                
            case 'years_months':
                ageText = `
                    <div class="age-item">
                        <span class="age-number">${age.years}</span>
                        <span class="age-label">${this.settings.labels.years}</span>
                    </div>
                    <div class="age-item">
                        <span class="age-number">${age.months}</span>
                        <span class="age-label">${this.settings.labels.months}</span>
                    </div>`;
                break;
                
            case 'full':
                ageText = `
                    <div class="age-item">
                        <span class="age-number">${age.years}</span>
                        <span class="age-label">${this.settings.labels.years}</span>
                    </div>
                    <div class="age-item">
                        <span class="age-number">${age.months}</span>
                        <span class="age-label">${this.settings.labels.months}</span>
                    </div>
                    <div class="age-item">
                        <span class="age-number">${age.days}</span>
                        <span class="age-label">${this.settings.labels.days}</span>
                    </div>`;
                break;
                
            case 'total_days':
                ageText = `<div class="age-item">
                    <span class="age-number">${age.totalDays.toLocaleString()}</span>
                    <span class="age-label">${this.settings.labels.days}</span>
                </div>`;
                break;
                
            case 'total_months':
                ageText = `<div class="age-item">
                    <span class="age-number">${age.totalMonths}</span>
                    <span class="age-label">${this.settings.labels.months}</span>
                </div>`;
                break;
        }

        this.ageDisplay.innerHTML = `<div class="age-container">${ageText}</div>`;

        // Show next birthday info
        if (this.settings.showNextBirthday && this.nextBirthdayInfo) {
            const birthdayText = age.daysUntilBirthday === 0 
                ? '🎉 Happy Birthday! 🎉' 
                : `${this.settings.labels.nextBirthday} <strong>${age.daysUntilBirthday} days</strong>`;
            
            this.nextBirthdayInfo.innerHTML = `<div class="next-birthday">${birthdayText}</div>`;
            this.nextBirthdayInfo.style.display = 'block';
        }

        // Show birth day info
        if (this.settings.showBirthDay && this.birthDayInfo) {
            this.birthDayInfo.innerHTML = `<div class="birth-day">${this.settings.labels.birthDay} <strong>${age.birthDayOfWeek}</strong></div>`;
            this.birthDayInfo.style.display = 'block';
        }
    }

    getDayOfWeek(date) {
        const days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
        return days[date.getDay()];
    }

    showResult() {
        this.resultContainer.style.display = 'block';
        this.resultContainer.classList.add('show');
        
        // Add animation
        setTimeout(() => {
            this.resultContainer.classList.add('animate');
        }, 10);
    }

    hideResult() {
        this.resultContainer.style.display = 'none';
        this.resultContainer.classList.remove('show', 'animate');
        
        if (this.nextBirthdayInfo) {
            this.nextBirthdayInfo.style.display = 'none';
        }
        
        if (this.birthDayInfo) {
            this.birthDayInfo.style.display = 'none';
        }
    }

    showError(message) {
        this.ageDisplay.innerHTML = `<div class="age-error">${message}</div>`;
        this.showResult();
        
        if (this.nextBirthdayInfo) {
            this.nextBirthdayInfo.style.display = 'none';
        }
        
        if (this.birthDayInfo) {
            this.birthDayInfo.style.display = 'none';
        }
    }

    // Public method to refresh calculator
    refresh() {
        this.hideResult();
        this.dateInput.value = '';
    }

    // Public method to set date programmatically
    setDate(dateString) {
        this.dateInput.value = dateString;
        this.calculateAge();
    }

    // Public method to get current age data
    getAgeData() {
        if (!this.dateInput.value) {
            return null;
        }
        
        const birthDate = new Date(this.dateInput.value);
        const today = new Date();
        
        return this.calculateDetailedAge(birthDate, today);
    }
}

// Initialize widgets when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    const widgets = document.querySelectorAll('.age-calculator-widget');
    
    widgets.forEach(widget => {
        new AgeCalculatorWidget(widget);
    });
});

// For Elementor editor compatibility
if (typeof elementorFrontend !== 'undefined') {
    elementorFrontend.hooks.addAction('frontend/element_ready/age_calculator.default', function($scope) {
        const widget = $scope.find('.age-calculator-widget')[0];
        if (widget) {
            new AgeCalculatorWidget(widget);
        }
    });
}
