/**
 * Business Chat Widget JavaScript
 * Handles WhatsApp, Messenger, Telegram and custom chat functionality
 */

class BusinessChatWidget {
    constructor(element) {
        this.element = element;
        this.button = element.querySelector('.chat-button');
        this.menu = element.querySelector('.chat-menu');
        this.tooltip = element.querySelector('.chat-tooltip');
        this.chatData = element.querySelector('.chat-data');
        
        this.position = element.dataset.position || 'bottom-right';
        this.enableAvailability = element.dataset.enableAvailability === 'true';
        this.timezone = element.dataset.timezone || 'UTC';
        this.offlineMessage = element.dataset.offlineMessage || 'We are currently offline.';
        
        this.isMenuOpen = false;
        this.availabilityHours = {};
        
        this.init();
    }
    
    init() {
        if (!this.button) return;
        
        this.parseAvailabilityHours();
        this.bindEvents();
        this.checkAvailability();
        this.showTooltip();
        
        // Auto-hide tooltip after delay
        setTimeout(() => {
            this.hideTooltip();
        }, 5000);
    }
    
    parseAvailabilityHours() {
        if (!this.enableAvailability) return;
        
        const days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
        days.forEach((day, index) => {
            const hours = this.element.dataset[day];
            if (hours) {
                const [start, end] = hours.split('-');
                this.availabilityHours[index + 1] = { start, end }; // 1-7 for Monday-Sunday
            }
        });
    }
    
    bindEvents() {
        this.button.addEventListener('click', (e) => {
            e.preventDefault();
            this.handleButtonClick();
        });
        
        // Handle menu item clicks
        if (this.menu) {
            const menuItems = this.menu.querySelectorAll('.chat-menu-item');
            menuItems.forEach(item => {
                item.addEventListener('click', (e) => {
                    e.preventDefault();
                    const platform = item.dataset.platform;
                    this.openChat(platform);
                    this.closeMenu();
                });
            });
        }
        
        // Close menu when clicking outside
        document.addEventListener('click', (e) => {
            if (this.isMenuOpen && !this.element.contains(e.target)) {
                this.closeMenu();
            }
        });
        
        // Keyboard support
        this.button.addEventListener('keydown', (e) => {
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                this.handleButtonClick();
            }
        });
        
        // Show/hide tooltip on hover
        this.button.addEventListener('mouseenter', () => {
            this.showTooltip();
        });
        
        this.button.addEventListener('mouseleave', () => {
            setTimeout(() => {
                if (!this.isMenuOpen) {
                    this.hideTooltip();
                }
            }, 300);
        });
    }
    
    handleButtonClick() {
        const platform = this.button.dataset.platform;
        
        if (platform === 'menu') {
            this.toggleMenu();
        } else {
            this.openChat(platform);
        }
    }
    
    toggleMenu() {
        if (this.isMenuOpen) {
            this.closeMenu();
        } else {
            this.openMenu();
        }
    }
    
    openMenu() {
        if (!this.menu) return;
        
        this.isMenuOpen = true;
        this.menu.style.display = 'block';
        
        // Animate menu items
        const items = this.menu.querySelectorAll('.chat-menu-item');
        items.forEach((item, index) => {
            setTimeout(() => {
                item.classList.add('show');
            }, index * 100);
        });
        
        this.hideTooltip();
        
        // Dispatch custom event
        this.element.dispatchEvent(new CustomEvent('chatMenuOpened', {
            detail: { position: this.position }
        }));
    }
    
    closeMenu() {
        if (!this.menu || !this.isMenuOpen) return;
        
        this.isMenuOpen = false;
        const items = this.menu.querySelectorAll('.chat-menu-item');
        
        items.forEach(item => {
            item.classList.remove('show');
        });
        
        setTimeout(() => {
            this.menu.style.display = 'none';
        }, 300);
        
        // Dispatch custom event
        this.element.dispatchEvent(new CustomEvent('chatMenuClosed', {
            detail: { position: this.position }
        }));
    }
    
    openChat(platform) {
        if (!this.isAvailable()) {
            this.showOfflineMessage();
            return;
        }
        
        const platformData = this.getPlatformData(platform);
        if (!platformData) return;
        
        let url = '';
        
        switch (platform) {
            case 'whatsapp':
                const number = platformData.number;
                const message = encodeURIComponent(platformData.message || '');
                url = `https://wa.me/${number}${message ? '?text=' + message : ''}`;
                break;
                
            case 'messenger':
                const username = platformData.username;
                url = `https://m.me/${username}`;
                break;
                
            case 'telegram':
                const telegramUsername = platformData.username;
                url = `https://t.me/${telegramUsername}`;
                break;
                
            case 'custom':
                url = platformData.url;
                break;
        }
        
        if (url) {
            const target = platformData.target || '_blank';
            if (target === '_blank') {
                window.open(url, '_blank', 'noopener,noreferrer');
            } else {
                window.location.href = url;
            }
            
            // Dispatch custom event
            this.element.dispatchEvent(new CustomEvent('chatOpened', {
                detail: { 
                    platform,
                    url,
                    available: this.isAvailable()
                }
            }));
        }
    }
    
    getPlatformData(platform) {
        if (!this.chatData) return null;
        
        const platformElement = this.chatData.querySelector(`[data-platform="${platform}"]`);
        if (!platformElement) return null;
        
        const data = {};
        const spans = platformElement.querySelectorAll('span');
        
        spans.forEach(span => {
            Object.keys(span.dataset).forEach(key => {
                data[key] = span.dataset[key];
            });
        });
        
        return data;
    }
    
    checkAvailability() {
        if (!this.enableAvailability) return;
        
        const isAvailable = this.isAvailable();
        this.element.classList.toggle('offline', !isAvailable);
        
        if (!isAvailable) {
            this.button.classList.add('offline');
        } else {
            this.button.classList.remove('offline');
        }
    }
    
    isAvailable() {
        if (!this.enableAvailability) return true;
        
        const now = new Date();
        const currentDay = now.getDay() || 7; // Convert Sunday (0) to 7
        const currentTime = now.toLocaleTimeString('en-US', { 
            hour12: false, 
            timeZone: this.timezone 
        }).slice(0, 5);
        
        const dayHours = this.availabilityHours[currentDay];
        if (!dayHours) return false;
        
        return currentTime >= dayHours.start && currentTime <= dayHours.end;
    }
    
    showOfflineMessage() {
        // Create temporary tooltip with offline message
        const offlineTooltip = document.createElement('div');
        offlineTooltip.className = 'chat-tooltip offline-tooltip';
        offlineTooltip.textContent = this.offlineMessage;
        
        this.element.appendChild(offlineTooltip);
        
        setTimeout(() => {
            offlineTooltip.classList.add('show');
        }, 10);
        
        setTimeout(() => {
            offlineTooltip.classList.remove('show');
            setTimeout(() => {
                if (offlineTooltip.parentNode) {
                    offlineTooltip.parentNode.removeChild(offlineTooltip);
                }
            }, 300);
        }, 3000);
    }
    
    showTooltip() {
        if (this.tooltip && !this.isMenuOpen) {
            this.tooltip.classList.add('show');
        }
    }
    
    hideTooltip() {
        if (this.tooltip) {
            this.tooltip.classList.remove('show');
        }
    }
    
    // Public methods
    openChatPlatform(platform) {
        this.openChat(platform);
    }
    
    toggleChatMenu() {
        this.toggleMenu();
    }
    
    updateAvailability() {
        this.checkAvailability();
    }
}

// Initialize widgets when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    initializeBusinessChatWidgets();
});

// Initialize for Elementor editor
jQuery(window).on('elementor/frontend/init', function() {
    elementorFrontend.hooks.addAction('frontend/element_ready/business_chat_widget.default', function($scope) {
        initializeBusinessChatWidgets($scope[0]);
    });
});

function initializeBusinessChatWidgets(container = document) {
    const widgets = container.querySelectorAll('.business-chat-widget');
    
    widgets.forEach(widget => {
        // Avoid double initialization
        if (widget.businessChatInstance) {
            return;
        }
        
        widget.businessChatInstance = new BusinessChatWidget(widget);
    });
}

// Handle dynamic content (AJAX, SPA)
if (window.MutationObserver) {
    const observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                mutation.addedNodes.forEach(function(node) {
                    if (node.nodeType === 1) { // Element node
                        const widgets = node.querySelectorAll ? node.querySelectorAll('.business-chat-widget') : [];
                        if (widgets.length > 0) {
                            initializeBusinessChatWidgets(node);
                        }
                        
                        if (node.classList && node.classList.contains('business-chat-widget')) {
                            initializeBusinessChatWidgets(node.parentElement);
                        }
                    }
                });
            }
        });
    });
    
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
}

// Global API
window.BusinessChatWidget = BusinessChatWidget;
window.initializeBusinessChatWidgets = initializeBusinessChatWidgets;

// Utility functions for developers
window.businessChat = {
    openWhatsApp: function(number, message) {
        const url = `https://wa.me/${number}${message ? '?text=' + encodeURIComponent(message) : ''}`;
        window.open(url, '_blank', 'noopener,noreferrer');
    },
    
    openMessenger: function(username) {
        const url = `https://m.me/${username}`;
        window.open(url, '_blank', 'noopener,noreferrer');
    },
    
    openTelegram: function(username) {
        const url = `https://t.me/${username}`;
        window.open(url, '_blank', 'noopener,noreferrer');
    },
    
    updateAvailability: function() {
        const widgets = document.querySelectorAll('.business-chat-widget');
        widgets.forEach(widget => {
            if (widget.businessChatInstance) {
                widget.businessChatInstance.updateAvailability();
            }
        });
    }
};
