<?php
/**
 * Business Chat Widget for Elementor
 * 
 * @package Relayout_Design
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Make sure Elementor is active
if (!class_exists('\Elementor\Widget_Base')) {
    return;
}

class Business_Chat_Widget extends \Elementor\Widget_Base {

    public function get_name() {
        return 'business_chat_widget';
    }

    public function get_title() {
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            return __('Business Chat Widget', 'relayout-design') . ' <span style="color: #ff6b35; font-weight: bold; font-size: 10px; background: #fff3f0; padding: 2px 6px; border-radius: 3px; margin-left: 5px;">PRO</span>';
        }
        return __('Business Chat Widget', 'relayout-design');
    }

    public function get_icon() {
        return 'eicon-comments';
    }

    public function get_categories() {
        return ['relayout-designs'];
    }

    public function get_keywords() {
        return ['chat', 'whatsapp', 'messenger', 'contact', 'floating', 'business'];
    }

    public function get_script_depends() {
        return ['business-chat-widget-script'];
    }

    public function get_style_depends() {
        return ['business-chat-widget-style'];
    }

    protected function register_controls() {
        // Content Section
        $this->start_controls_section(
            'content_section',
            [
                'label' => __('Chat Settings', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'chat_platforms',
            [
                'label' => __('Chat Platforms', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT2,
                'multiple' => true,
                'default' => ['whatsapp'],
                'options' => [
                    'whatsapp' => __('WhatsApp', 'relayout-design'),
                    'messenger' => __('Facebook Messenger', 'relayout-design'),
                    'telegram' => __('Telegram', 'relayout-design'),
                    'custom' => __('Custom Link', 'relayout-design'),
                ],
                'description' => __('Select which chat platforms to enable', 'relayout-design'),
            ]
        );

        $this->add_control(
            'whatsapp_number',
            [
                'label' => __('WhatsApp Number', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'placeholder' => __('1234567890', 'relayout-design'),
                'description' => __('Enter phone number with country code (without + or spaces)', 'relayout-design'),
                'condition' => [
                    'chat_platforms' => 'whatsapp',
                ],
            ]
        );

        $this->add_control(
            'whatsapp_message',
            [
                'label' => __('WhatsApp Default Message', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => __('Hello! I would like to get in touch.', 'relayout-design'),
                'placeholder' => __('Enter default message...', 'relayout-design'),
                'condition' => [
                    'chat_platforms' => 'whatsapp',
                ],
            ]
        );

        $this->add_control(
            'messenger_username',
            [
                'label' => __('Messenger Username', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'placeholder' => __('your.page.name', 'relayout-design'),
                'description' => __('Enter your Facebook page username', 'relayout-design'),
                'condition' => [
                    'chat_platforms' => 'messenger',
                ],
            ]
        );

        $this->add_control(
            'telegram_username',
            [
                'label' => __('Telegram Username', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'placeholder' => __('yourusername', 'relayout-design'),
                'description' => __('Enter your Telegram username (without @)', 'relayout-design'),
                'condition' => [
                    'chat_platforms' => 'telegram',
                ],
            ]
        );

        $this->add_control(
            'custom_link',
            [
                'label' => __('Custom Chat Link', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::URL,
                'placeholder' => __('https://your-chat-link.com', 'relayout-design'),
                'condition' => [
                    'chat_platforms' => 'custom',
                ],
            ]
        );

        $this->add_control(
            'custom_label',
            [
                'label' => __('Custom Chat Label', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Chat', 'relayout-design'),
                'condition' => [
                    'chat_platforms' => 'custom',
                ],
            ]
        );

        $this->end_controls_section();

        // Position Section
        $this->start_controls_section(
            'position_section',
            [
                'label' => __('Position & Display', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'position',
            [
                'label' => __('Position', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'bottom-right',
                'options' => [
                    'bottom-left' => __('Bottom Left', 'relayout-design'),
                    'bottom-right' => __('Bottom Right', 'relayout-design'),
                    'top-left' => __('Top Left', 'relayout-design'),
                    'top-right' => __('Top Right', 'relayout-design'),
                ],
            ]
        );

        $this->add_responsive_control(
            'offset_horizontal',
            [
                'label' => __('Horizontal Offset', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 20,
                ],
            ]
        );

        $this->add_responsive_control(
            'offset_vertical',
            [
                'label' => __('Vertical Offset', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 20,
                ],
            ]
        );

        $this->add_control(
            'show_tooltip',
            [
                'label' => __('Show Tooltip', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'relayout-design'),
                'label_off' => __('Hide', 'relayout-design'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'tooltip_text',
            [
                'label' => __('Tooltip Text', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Need help? Chat with us!', 'relayout-design'),
                'condition' => [
                    'show_tooltip' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        // Availability Section
        $this->start_controls_section(
            'availability_section',
            [
                'label' => __('Availability Hours', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'enable_availability',
            [
                'label' => __('Enable Availability Hours', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'relayout-design'),
                'label_off' => __('No', 'relayout-design'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'timezone',
            [
                'label' => __('Timezone', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'UTC',
                'options' => [
                    'UTC' => 'UTC',
                    'America/New_York' => 'Eastern Time',
                    'America/Chicago' => 'Central Time',
                    'America/Denver' => 'Mountain Time',
                    'America/Los_Angeles' => 'Pacific Time',
                    'Europe/London' => 'London',
                    'Europe/Paris' => 'Paris',
                    'Asia/Tokyo' => 'Tokyo',
                    'Asia/Kolkata' => 'India',
                    'Australia/Sydney' => 'Sydney',
                ],
                'condition' => [
                    'enable_availability' => 'yes',
                ],
            ]
        );

        $days = [
            'monday' => __('Monday', 'relayout-design'),
            'tuesday' => __('Tuesday', 'relayout-design'),
            'wednesday' => __('Wednesday', 'relayout-design'),
            'thursday' => __('Thursday', 'relayout-design'),
            'friday' => __('Friday', 'relayout-design'),
            'saturday' => __('Saturday', 'relayout-design'),
            'sunday' => __('Sunday', 'relayout-design'),
        ];

        foreach ($days as $day => $label) {
            $this->add_control(
                $day . '_enabled',
                [
                    'label' => $label,
                    'type' => \Elementor\Controls_Manager::SWITCHER,
                    'label_on' => __('Open', 'relayout-design'),
                    'label_off' => __('Closed', 'relayout-design'),
                    'return_value' => 'yes',
                    'default' => 'yes',
                    'condition' => [
                        'enable_availability' => 'yes',
                    ],
                ]
            );

            $this->add_control(
                $day . '_start',
                [
                    'label' => $label . ' - ' . __('Start Time', 'relayout-design'),
                    'type' => \Elementor\Controls_Manager::TEXT,
                    'default' => '09:00',
                    'placeholder' => '09:00',
                    'condition' => [
                        'enable_availability' => 'yes',
                        $day . '_enabled' => 'yes',
                    ],
                ]
            );

            $this->add_control(
                $day . '_end',
                [
                    'label' => $label . ' - ' . __('End Time', 'relayout-design'),
                    'type' => \Elementor\Controls_Manager::TEXT,
                    'default' => '17:00',
                    'placeholder' => '17:00',
                    'condition' => [
                        'enable_availability' => 'yes',
                        $day . '_enabled' => 'yes',
                    ],
                ]
            );
        }

        $this->add_control(
            'offline_message',
            [
                'label' => __('Offline Message', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => __('We are currently offline. Please leave a message and we will get back to you.', 'relayout-design'),
                'condition' => [
                    'enable_availability' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        // Button Style Section
        $this->start_controls_section(
            'button_style_section',
            [
                'label' => __('Button Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'button_size',
            [
                'label' => __('Button Size', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 40,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 60,
                ],
                'selectors' => [
                    '{{WRAPPER}} .chat-button' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'custom_icon',
            [
                'label' => __('Custom Icon', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'description' => __('Upload custom icon (optional)', 'relayout-design'),
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label' => __('Icon Size', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 20,
                        'max' => 60,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 30,
                ],
                'selectors' => [
                    '{{WRAPPER}} .chat-button .chat-icon' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .chat-button .chat-icon img' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs('button_style_tabs');

        $this->start_controls_tab(
            'button_normal_tab',
            [
                'label' => __('Normal', 'relayout-design'),
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'button_background',
                'label' => __('Background', 'relayout-design'),
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .chat-button',
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                    'color' => [
                        'default' => '#25D366',
                    ],
                ],
            ]
        );

        $this->add_control(
            'button_icon_color',
            [
                'label' => __('Icon Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .chat-button .chat-icon' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_hover_tab',
            [
                'label' => __('Hover', 'relayout-design'),
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'button_hover_background',
                'label' => __('Background', 'relayout-design'),
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .chat-button:hover',
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                    'color' => [
                        'default' => '#128C7E',
                    ],
                ],
            ]
        );

        $this->add_control(
            'button_hover_icon_color',
            [
                'label' => __('Icon Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .chat-button:hover .chat-icon' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'hover_animation',
            [
                'label' => __('Hover Animation', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'pulse',
                'options' => [
                    'none' => __('None', 'relayout-design'),
                    'pulse' => __('Pulse', 'relayout-design'),
                    'bounce' => __('Bounce', 'relayout-design'),
                    'shake' => __('Shake', 'relayout-design'),
                    'scale' => __('Scale', 'relayout-design'),
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'button_border',
                'label' => __('Border', 'relayout-design'),
                'selector' => '{{WRAPPER}} .chat-button',
            ]
        );

        $this->add_responsive_control(
            'button_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 50,
                    'right' => 50,
                    'bottom' => 50,
                    'left' => 50,
                    'unit' => '%',
                ],
                'selectors' => [
                    '{{WRAPPER}} .chat-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'button_box_shadow',
                'label' => __('Box Shadow', 'relayout-design'),
                'selector' => '{{WRAPPER}} .chat-button',
                'fields_options' => [
                    'box_shadow_type' => [
                        'default' => 'yes',
                    ],
                    'box_shadow' => [
                        'default' => [
                            'horizontal' => 0,
                            'vertical' => 4,
                            'blur' => 15,
                            'spread' => 0,
                            'color' => 'rgba(0,0,0,0.2)',
                        ],
                    ],
                ],
            ]
        );

        $this->end_controls_section();

        // Tooltip Style Section
        $this->start_controls_section(
            'tooltip_style_section',
            [
                'label' => __('Tooltip Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_tooltip' => 'yes',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'tooltip_typography',
                'label' => __('Typography', 'relayout-design'),
                'selector' => '{{WRAPPER}} .chat-tooltip',
            ]
        );

        $this->add_control(
            'tooltip_text_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .chat-tooltip' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'tooltip_background_color',
            [
                'label' => __('Background Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}} .chat-tooltip' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .chat-tooltip::after' => 'border-top-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'tooltip_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'default' => [
                    'top' => 8,
                    'right' => 12,
                    'bottom' => 8,
                    'left' => 12,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .chat-tooltip' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'tooltip_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 4,
                    'right' => 4,
                    'bottom' => 4,
                    'left' => 4,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .chat-tooltip' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        // Check if premium features are available
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            $upgrade_prompt = relayout_designs_show_upgrade_prompt('Business Chat Widget');
            if ($upgrade_prompt) {
                echo $upgrade_prompt;
                return;
            }
        }
        
        $settings = $this->get_settings_for_display();
        
        $chat_platforms = $settings['chat_platforms'];
        $position = $settings['position'];
        $show_tooltip = $settings['show_tooltip'] === 'yes';
        $enable_availability = $settings['enable_availability'] === 'yes';
        $hover_animation = $settings['hover_animation'];
        
        $this->add_render_attribute('wrapper', 'class', 'business-chat-widget');
        $this->add_render_attribute('wrapper', 'class', 'position-' . $position);
        $this->add_render_attribute('wrapper', 'data-position', $position);
        $this->add_render_attribute('wrapper', 'data-enable-availability', $enable_availability ? 'true' : 'false');
        
        if ($enable_availability) {
            $this->add_render_attribute('wrapper', 'data-timezone', $settings['timezone']);
            $this->add_render_attribute('wrapper', 'data-offline-message', $settings['offline_message']);
            
            $days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
            foreach ($days as $day) {
                if ($settings[$day . '_enabled'] === 'yes') {
                    $this->add_render_attribute('wrapper', 'data-' . $day, $settings[$day . '_start'] . '-' . $settings[$day . '_end']);
                }
            }
        }
        
        $this->add_render_attribute('button', 'class', 'chat-button');
        if ($hover_animation !== 'none') {
            $this->add_render_attribute('button', 'class', 'hover-' . $hover_animation);
        }
        
        // Set position styles
        $horizontal_offset = $settings['offset_horizontal']['size'] . 'px';
        $vertical_offset = $settings['offset_vertical']['size'] . 'px';
        
        $position_styles = '';
        switch ($position) {
            case 'bottom-left':
                $position_styles = "bottom: {$vertical_offset}; left: {$horizontal_offset};";
                break;
            case 'bottom-right':
                $position_styles = "bottom: {$vertical_offset}; right: {$horizontal_offset};";
                break;
            case 'top-left':
                $position_styles = "top: {$vertical_offset}; left: {$horizontal_offset};";
                break;
            case 'top-right':
                $position_styles = "top: {$vertical_offset}; right: {$horizontal_offset};";
                break;
        }
        
        $this->add_render_attribute('wrapper', 'style', $position_styles);
        
        ?>
        <div <?php echo $this->get_render_attribute_string('wrapper'); ?>>
            <?php if (count($chat_platforms) === 1) : ?>
                <!-- Single platform button -->
                <?php $platform = $chat_platforms[0]; ?>
                <button <?php echo $this->get_render_attribute_string('button'); ?> data-platform="<?php echo esc_attr($platform); ?>">
                    <span class="chat-icon">
                        <?php if (!empty($settings['custom_icon']['url'])) : ?>
                            <img src="<?php echo esc_url($settings['custom_icon']['url']); ?>" alt="Chat">
                        <?php else : ?>
                            <?php echo $this->get_platform_icon($platform); ?>
                        <?php endif; ?>
                    </span>
                </button>
            <?php else : ?>
                <!-- Multiple platforms - show menu -->
                <button <?php echo $this->get_render_attribute_string('button'); ?> data-platform="menu">
                    <span class="chat-icon">
                        <?php if (!empty($settings['custom_icon']['url'])) : ?>
                            <img src="<?php echo esc_url($settings['custom_icon']['url']); ?>" alt="Chat">
                        <?php else : ?>
                            <i class="fas fa-comments"></i>
                        <?php endif; ?>
                    </span>
                </button>
                
                <div class="chat-menu" style="display: none;">
                    <?php foreach ($chat_platforms as $platform) : ?>
                        <button class="chat-menu-item" data-platform="<?php echo esc_attr($platform); ?>">
                            <span class="chat-icon"><?php echo $this->get_platform_icon($platform); ?></span>
                            <span class="chat-label"><?php echo $this->get_platform_label($platform, $settings); ?></span>
                        </button>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
            
            <?php if ($show_tooltip) : ?>
                <div class="chat-tooltip"><?php echo esc_html($settings['tooltip_text']); ?></div>
            <?php endif; ?>
            
            <!-- Hidden data for JavaScript -->
            <div class="chat-data" style="display: none;">
                <?php foreach ($chat_platforms as $platform) : ?>
                    <div data-platform="<?php echo esc_attr($platform); ?>">
                        <?php if ($platform === 'whatsapp') : ?>
                            <span data-number="<?php echo esc_attr($settings['whatsapp_number']); ?>"></span>
                            <span data-message="<?php echo esc_attr($settings['whatsapp_message']); ?>"></span>
                        <?php elseif ($platform === 'messenger') : ?>
                            <span data-username="<?php echo esc_attr($settings['messenger_username']); ?>"></span>
                        <?php elseif ($platform === 'telegram') : ?>
                            <span data-username="<?php echo esc_attr($settings['telegram_username']); ?>"></span>
                        <?php elseif ($platform === 'custom') : ?>
                            <span data-url="<?php echo esc_url($settings['custom_link']['url']); ?>"></span>
                            <span data-target="<?php echo $settings['custom_link']['is_external'] ? '_blank' : '_self'; ?>"></span>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
    }
    
    private function get_platform_icon($platform) {
        $icons = [
            'whatsapp' => '<i class="fab fa-whatsapp"></i>',
            'messenger' => '<i class="fab fa-facebook-messenger"></i>',
            'telegram' => '<i class="fab fa-telegram-plane"></i>',
            'custom' => '<i class="fas fa-comments"></i>',
        ];
        
        return isset($icons[$platform]) ? $icons[$platform] : '<i class="fas fa-comments"></i>';
    }
    
    private function get_platform_label($platform, $settings) {
        $labels = [
            'whatsapp' => __('WhatsApp', 'relayout-design'),
            'messenger' => __('Messenger', 'relayout-design'),
            'telegram' => __('Telegram', 'relayout-design'),
            'custom' => $settings['custom_label'] ?: __('Chat', 'relayout-design'),
        ];
        
        return isset($labels[$platform]) ? $labels[$platform] : __('Chat', 'relayout-design');
    }
}
