/**
 * Business Hours Widget JavaScript
 * Handles real-time status updates and timezone calculations
 */

class BusinessHoursWidget {
    constructor(element) {
        this.element = element;
        this.statusElement = element.querySelector('.business-status');
        this.hoursRows = element.querySelectorAll('.hours-row');
        this.intervalId = null;
        
        this.init();
    }

    init() {
        // Clear any existing interval
        if (this.intervalId) {
            clearInterval(this.intervalId);
        }
        
        // Refresh data from DOM
        this.refreshData();
        
        this.updateStatus();
        this.highlightToday();
        this.checkSpecialHours();
        
        // Update status every minute
        this.intervalId = setInterval(() => {
            this.refreshData();
            this.updateStatus();
        }, 60000);
    }

    refreshData() {
        // Always get fresh data from DOM attributes
        this.timezone = this.element.dataset.timezone;
        this.hours = JSON.parse(this.element.dataset.hours || '{}');
        this.specialHours = JSON.parse(this.element.dataset.special || '{}');
        
        // Refresh DOM elements in case they changed
        this.statusElement = this.element.querySelector('.business-status');
        this.hoursRows = this.element.querySelectorAll('.hours-row');
    }

    getCurrentTime() {
        const now = new Date();
        
        // Ensure timezone is valid, fallback to system timezone
        let timeInTimezone;
        try {
            if (this.timezone) {
                timeInTimezone = new Date(now.toLocaleString("en-US", {timeZone: this.timezone}));
            } else {
                timeInTimezone = now;
            }
        } catch (error) {
            console.warn('Invalid timezone:', this.timezone, 'Using system timezone');
            timeInTimezone = now;
        }
        
        return {
            date: timeInTimezone,
            dayOfWeek: timeInTimezone.getDay(), // 0 = Sunday, 1 = Monday, etc.
            timeString: timeInTimezone.toTimeString().slice(0, 5), // HH:MM format
            dateString: timeInTimezone.toISOString().slice(0, 10) // YYYY-MM-DD format
        };
    }

    getDayKey(dayOfWeek) {
        const days = ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'];
        return days[dayOfWeek];
    }

    isTimeInRange(currentTime, openTime, closeTime) {
        const current = this.timeToMinutes(currentTime);
        const open = this.timeToMinutes(openTime);
        const close = this.timeToMinutes(closeTime);
        
        // Handle overnight hours (e.g., 22:00 - 02:00)
        if (close < open) {
            return current >= open || current <= close;
        }
        
        return current >= open && current <= close;
    }

    timeToMinutes(timeString) {
        const [hours, minutes] = timeString.split(':').map(Number);
        return hours * 60 + minutes;
    }

    updateStatus() {
        if (!this.statusElement) return;

        const current = this.getCurrentTime();
        const dayKey = this.getDayKey(current.dayOfWeek);
        
        let isOpen = false;
        let statusText = '';
        
        // Get status text from data attributes (fresh from DOM)
        const openText = this.statusElement.dataset.openText || "We're Open";
        const closedText = this.statusElement.dataset.closedText || "We're Closed";
        
        console.log('Debug - Current time:', current);
        console.log('Debug - Day key:', dayKey);
        console.log('Debug - Hours data:', this.hours);
        console.log('Debug - Special hours:', this.specialHours);
        
        // Check for special hours first
        if (this.specialHours && this.specialHours[current.dateString]) {
            const special = this.specialHours[current.dateString];
            console.log('Debug - Special hours found:', special);
            if (special.type === 'closed') {
                isOpen = false;
                statusText = closedText + ` (${special.label})`;
            } else if (special.type === 'custom') {
                isOpen = this.isTimeInRange(current.timeString, special.open, special.close);
                statusText = isOpen ? 
                    openText + ` (${special.label})` : 
                    closedText + ` (${special.label})`;
            }
        } else if (this.hours && this.hours[dayKey]) {
            // Check regular business hours
            const dayHours = this.hours[dayKey];
            console.log('Debug - Day hours:', dayHours);
            console.log('Debug - Time check:', current.timeString, 'between', dayHours.open, 'and', dayHours.close);
            isOpen = this.isTimeInRange(current.timeString, dayHours.open, dayHours.close);
            console.log('Debug - Is open:', isOpen);
            statusText = isOpen ? openText : closedText;
        } else {
            // Day is closed
            console.log('Debug - No hours found for day, marking as closed');
            isOpen = false;
            statusText = closedText;
        }

        console.log('Debug - Final status:', statusText, isOpen);
        
        // Update status display
        this.statusElement.textContent = statusText;
        this.statusElement.className = `business-status ${isOpen ? 'open' : 'closed'}`;
    }

    highlightToday() {
        const current = this.getCurrentTime();
        const dayKey = this.getDayKey(current.dayOfWeek);
        
        this.hoursRows.forEach(row => {
            row.classList.remove('today');
            if (row.dataset.day === dayKey) {
                row.classList.add('today');
            }
        });
    }

    checkSpecialHours() {
        const current = this.getCurrentTime();
        
        this.hoursRows.forEach(row => {
            row.classList.remove('special');
            
            if (this.specialHours[current.dateString]) {
                const special = this.specialHours[current.dateString];
                const dayKey = this.getDayKey(current.dayOfWeek);
                
                if (row.dataset.day === dayKey) {
                    row.classList.add('special');
                    const timeElement = row.querySelector('.time');
                    
                    if (special.type === 'closed') {
                        timeElement.textContent = `Closed (${special.label})`;
                    } else if (special.type === 'custom') {
                        timeElement.textContent = `${special.open} - ${special.close} (${special.label})`;
                    }
                }
            }
        });
    }

    // Public method to manually refresh the widget
    refresh() {
        this.refreshData();
        this.updateStatus();
        this.highlightToday();
        this.checkSpecialHours();
    }

    // Cleanup method to prevent memory leaks
    destroy() {
        if (this.intervalId) {
            clearInterval(this.intervalId);
            this.intervalId = null;
        }
    }
}

// Store widget instances to prevent memory leaks
const widgetInstances = new WeakMap();

// Initialize all business hours widgets when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    const widgets = document.querySelectorAll('.business-hours-widget');
    
    widgets.forEach(widget => {
        if (!widgetInstances.has(widget)) {
            const instance = new BusinessHoursWidget(widget);
            widgetInstances.set(widget, instance);
        }
    });
});

// Re-initialize widgets after Elementor preview updates
if (typeof elementorFrontend !== 'undefined') {
    // Handle Elementor editor mode
    elementorFrontend.hooks.addAction('frontend/element_ready/business_hours.default', function($scope) {
        const widget = $scope.find('.business-hours-widget')[0];
        if (widget) {
            // Destroy existing instance if it exists
            if (widgetInstances.has(widget)) {
                const existingInstance = widgetInstances.get(widget);
                existingInstance.destroy();
                widgetInstances.delete(widget);
            }
            
            // Small delay to ensure DOM is fully updated
            setTimeout(() => {
                const instance = new BusinessHoursWidget(widget);
                widgetInstances.set(widget, instance);
            }, 50);
        }
    });

    // Handle Elementor editor changes
    if (typeof elementor !== 'undefined') {
        // Listen for model changes
        elementor.hooks.addAction('panel/open_editor/widget/business_hours', function(panel, model, view) {
            // Listen for setting changes
            model.on('change', function() {
                setTimeout(() => {
                    const widget = view.$el.find('.business-hours-widget')[0];
                    if (widget && widgetInstances.has(widget)) {
                        const instance = widgetInstances.get(widget);
                        instance.refresh();
                    }
                }, 100);
            });
        });
    }
}

// Expose BusinessHoursWidget globally for manual initialization
window.BusinessHoursWidget = BusinessHoursWidget;
