<?php
/**
 * Business Hours Widget for Elementor
 *
 * @package RelayoutDesigns
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Make sure Elementor is active
if (!class_exists('\Elementor\Widget_Base')) {
    return;
}

class Business_Hours_Widget extends \Elementor\Widget_Base {

    public function get_name() {
        return 'business_hours';
    }

    public function get_title() {
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            return __('Business Hours', 'relayout-design') . ' <span style="color: #ff6b35; font-weight: bold; font-size: 10px; background: #fff3f0; padding: 2px 6px; border-radius: 3px; margin-left: 5px;">PRO</span>';
        }
        return __('Business Hours', 'relayout-design');
    }

    public function get_icon() {
        return 'eicon-clock-o';
    }

    public function get_categories() {
        return ['relayout-designs'];
    }

    public function get_script_depends() {
        return ['business-hours-widget-script'];
    }

    public function get_style_depends() {
        return ['business-hours-widget-style'];
    }

    protected function register_controls() {
        // General Settings Section
        $this->start_controls_section(
            'general_section',
            [
                'label' => __('General Settings', 'relayout-design'),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'timezone',
            [
                'label' => __('Timezone', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'America/New_York',
                'options' => [
                    'America/New_York' => __('Eastern Time (ET)', 'relayout-design'),
                    'America/Chicago' => __('Central Time (CT)', 'relayout-design'),
                    'America/Denver' => __('Mountain Time (MT)', 'relayout-design'),
                    'America/Los_Angeles' => __('Pacific Time (PT)', 'relayout-design'),
                    'Europe/London' => __('London (GMT)', 'relayout-design'),
                    'Europe/Paris' => __('Paris (CET)', 'relayout-design'),
                    'Asia/Tokyo' => __('Tokyo (JST)', 'relayout-design'),
                    'Asia/Kolkata' => __('India (IST)', 'relayout-design'),
                    'Australia/Sydney' => __('Sydney (AEST)', 'relayout-design'),
                ],
            ]
        );

        $this->add_control(
            'show_status',
            [
                'label' => __('Show Open/Closed Status', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
                'description' => __('Note: Real-time status updates will only display on published pages or preview mode, not in the Elementor editor.', 'relayout-design'),
            ]
        );

        $this->add_control(
            'status_text_open',
            [
                'label' => __('Open Status Text', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('We\'re Open', 'relayout-design'),
                'condition' => [
                    'show_status' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'status_text_closed',
            [
                'label' => __('Closed Status Text', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('We\'re Closed', 'relayout-design'),
                'condition' => [
                    'show_status' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        // Business Hours Section
        $this->start_controls_section(
            'hours_section',
            [
                'label' => __('Business Hours', 'relayout-design'),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $days = [
            'monday' => __('Monday', 'relayout-design'),
            'tuesday' => __('Tuesday', 'relayout-design'),
            'wednesday' => __('Wednesday', 'relayout-design'),
            'thursday' => __('Thursday', 'relayout-design'),
            'friday' => __('Friday', 'relayout-design'),
            'saturday' => __('Saturday', 'relayout-design'),
            'sunday' => __('Sunday', 'relayout-design'),
        ];

        foreach ($days as $day_key => $day_label) {
            $this->add_control(
                $day_key . '_enabled',
                [
                    'label' => $day_label . ' - ' . __('Open', 'relayout-design'),
                    'type' => \Elementor\Controls_Manager::SWITCHER,
                    'default' => $day_key !== 'sunday' ? 'yes' : '',
                ]
            );

            $this->add_control(
                $day_key . '_open',
                [
                    'label' => $day_label . ' - ' . __('Opening Time', 'relayout-design'),
                    'type' => \Elementor\Controls_Manager::TEXT,
                    'default' => '09:00',
                    'placeholder' => '09:00',
                    'condition' => [
                        $day_key . '_enabled' => 'yes',
                    ],
                ]
            );

            $this->add_control(
                $day_key . '_close',
                [
                    'label' => $day_label . ' - ' . __('Closing Time', 'relayout-design'),
                    'type' => \Elementor\Controls_Manager::TEXT,
                    'default' => '17:00',
                    'placeholder' => '17:00',
                    'condition' => [
                        $day_key . '_enabled' => 'yes',
                    ],
                ]
            );
        }

        $this->end_controls_section();

        // Special Hours Section
        $this->start_controls_section(
            'special_hours_section',
            [
                'label' => __('Special Hours & Holidays', 'relayout-design'),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'special_date',
            [
                'label' => __('Date', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DATE_TIME,
                'picker_options' => [
                    'dateFormat' => 'Y-m-d',
                    'enableTime' => false,
                ],
            ]
        );

        $repeater->add_control(
            'special_label',
            [
                'label' => __('Label', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Holiday', 'relayout-design'),
            ]
        );

        $repeater->add_control(
            'special_type',
            [
                'label' => __('Type', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'closed' => __('Closed', 'relayout-design'),
                    'custom' => __('Custom Hours', 'relayout-design'),
                ],
                'default' => 'closed',
            ]
        );

        $repeater->add_control(
            'special_open',
            [
                'label' => __('Opening Time', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => '10:00',
                'condition' => [
                    'special_type' => 'custom',
                ],
            ]
        );

        $repeater->add_control(
            'special_close',
            [
                'label' => __('Closing Time', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => '15:00',
                'condition' => [
                    'special_type' => 'custom',
                ],
            ]
        );

        $this->add_control(
            'special_hours',
            [
                'label' => __('Special Hours', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => '{{{ special_label }}} - {{{ special_date }}}',
            ]
        );

        $this->end_controls_section();

        // Style Section - Status
        $this->start_controls_section(
            'status_style_section',
            [
                'label' => __('Status Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'open_status_color',
            [
                'label' => __('Open Status Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#28a745',
                'selectors' => [
                    '{{WRAPPER}} .business-status.open' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'closed_status_color',
            [
                'label' => __('Closed Status Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#dc3545',
                'selectors' => [
                    '{{WRAPPER}} .business-status.closed' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'status_typography',
                'selector' => '{{WRAPPER}} .business-status',
            ]
        );

        $this->end_controls_section();

        // Style Section - Hours
        $this->start_controls_section(
            'hours_style_section',
            [
                'label' => __('Hours Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'day_color',
            [
                'label' => __('Day Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .business-hours .day' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'time_color',
            [
                'label' => __('Time Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .business-hours .time' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'today_highlight_color',
            [
                'label' => __('Today Highlight Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#007cba',
                'selectors' => [
                    '{{WRAPPER}} .business-hours .today' => 'background-color: {{VALUE}}; color: #fff;',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'hours_typography',
                'selector' => '{{WRAPPER}} .business-hours',
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        // Check if premium features are available
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            $upgrade_prompt = relayout_designs_show_upgrade_prompt('Business Hours Widget');
            if ($upgrade_prompt) {
                echo $upgrade_prompt;
                return;
            }
        }
        
        $settings = $this->get_settings_for_display();
        
        $days = [
            'monday' => __('Monday', 'relayout-design'),
            'tuesday' => __('Tuesday', 'relayout-design'),
            'wednesday' => __('Wednesday', 'relayout-design'),
            'thursday' => __('Thursday', 'relayout-design'),
            'friday' => __('Friday', 'relayout-design'),
            'saturday' => __('Saturday', 'relayout-design'),
            'sunday' => __('Sunday', 'relayout-design'),
        ];

        // Prepare business hours data
        $business_hours = [];
        foreach ($days as $day_key => $day_label) {
            if ($settings[$day_key . '_enabled'] === 'yes') {
                $business_hours[$day_key] = [
                    'open' => $settings[$day_key . '_open'],
                    'close' => $settings[$day_key . '_close'],
                ];
            }
        }

        // Prepare special hours data
        $special_hours = [];
        if (!empty($settings['special_hours'])) {
            foreach ($settings['special_hours'] as $special) {
                $date = date('Y-m-d', strtotime($special['special_date']));
                $special_hours[$date] = [
                    'label' => $special['special_label'],
                    'type' => $special['special_type'],
                    'open' => $special['special_type'] === 'custom' ? $special['special_open'] : null,
                    'close' => $special['special_type'] === 'custom' ? $special['special_close'] : null,
                ];
            }
        }
        ?>
        <div class="business-hours-widget" 
             data-timezone="<?php echo esc_attr($settings['timezone']); ?>"
             data-hours="<?php echo esc_attr(json_encode($business_hours)); ?>"
             data-special="<?php echo esc_attr(json_encode($special_hours)); ?>">
            
            <?php if ($settings['show_status'] === 'yes'): ?>
                <div class="business-status" 
                     data-open-text="<?php echo esc_attr($settings['status_text_open']); ?>"
                     data-closed-text="<?php echo esc_attr($settings['status_text_closed']); ?>">
                    <?php echo esc_html($settings['status_text_closed']); ?>
                </div>
            <?php endif; ?>

            <div class="business-hours">
                <?php foreach ($days as $day_key => $day_label): ?>
                    <div class="hours-row" data-day="<?php echo esc_attr($day_key); ?>">
                        <span class="day"><?php echo esc_html($day_label); ?></span>
                        <span class="time">
                            <?php if ($settings[$day_key . '_enabled'] === 'yes'): ?>
                                <?php echo esc_html($settings[$day_key . '_open'] . ' - ' . $settings[$day_key . '_close']); ?>
                            <?php else: ?>
                                <?php echo esc_html__('Closed', 'relayout-design'); ?>
                            <?php endif; ?>
                        </span>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
    }
}
