document.addEventListener('DOMContentLoaded', function() {
    // Initialize checklist managers
    initializeChecklistManagers();
    
    function initializeChecklistManagers() {
        const checklistWrappers = document.querySelectorAll('.checklist-manager-wrapper');
        
        checklistWrappers.forEach(wrapper => {
            const widgetId = wrapper.getAttribute('data-widget-id');
            const checkboxes = wrapper.querySelectorAll('.checklist-checkbox');
            
            // Add event listeners to all checkboxes in this widget
            checkboxes.forEach(checkbox => {
                checkbox.addEventListener('change', function() {
                    updateRelayoutChecklistState(widgetId, checkboxes);
                });
            });
            
            // Initial state check
            updateRelayoutChecklistState(widgetId, checkboxes);
        });
    }
    
    function updateRelayoutChecklistState(widgetId, checkboxes) {
        const relayoutChecklistSections = document.querySelectorAll('.relayout_checklist');
        
        if (relayoutChecklistSections.length === 0) {
            return;
        }
        
        // Count checked checkboxes
        const checkedCount = Array.from(checkboxes).filter(checkbox => checkbox.checked).length;
        const totalCount = checkboxes.length;
        
        // Check if all checkboxes are checked
        const allChecked = checkedCount === totalCount && totalCount > 0;
        
        // Update all .relayout_checklist sections
        relayoutChecklistSections.forEach(section => {
            if (allChecked) {
                // Enable the section
                section.classList.remove('disabled');
                section.classList.add('enabled');
            } else {
                // Disable the section
                section.classList.remove('enabled');
                section.classList.add('disabled');
            }
        });
        
        // Trigger custom event for developers
        const event = new CustomEvent('checklistStateChanged', {
            detail: {
                widgetId: widgetId,
                checkedCount: checkedCount,
                totalCount: totalCount,
                allChecked: allChecked
            }
        });
        document.dispatchEvent(event);
    }
    
    // Handle dynamic content loading (for AJAX/SPA compatibility)
    const observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            if (mutation.type === 'childList') {
                mutation.addedNodes.forEach(function(node) {
                    if (node.nodeType === 1) { // Element node
                        const newChecklistWrappers = node.querySelectorAll ? 
                            node.querySelectorAll('.checklist-manager-wrapper') : [];
                        
                        if (newChecklistWrappers.length > 0) {
                            initializeChecklistManagers();
                        }
                    }
                });
            }
        });
    });
    
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
});
