<?php
/**
 * Checklist Manager Widget for Elementor
 *
 * @package RelayoutDesigns
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Make sure Elementor is active
if (!class_exists('\Elementor\Widget_Base')) {
    return;
}

class Checklist_Manager_Widget extends \Elementor\Widget_Base {

    public function get_name() {
        return 'checklist_manager';
    }

    public function get_title() {
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            return __('Checklist Manager', 'relayout-design') . ' <span style="color: #ff6b35; font-weight: bold; font-size: 10px; background: #fff3f0; padding: 2px 6px; border-radius: 3px; margin-left: 5px;">PRO</span>';
        }
        return __('Checklist Manager', 'relayout-design');
    }

    public function get_icon() {
        return 'eicon-editor-list-ul';
    }

    public function get_categories() {
        return ['relayout-designs'];
    }

    public function get_script_depends() {
        return ['checklist-manager-widget-script'];
    }

    public function get_style_depends() {
        return ['checklist-manager-widget-style'];
    }

    protected function register_controls() {
        // Content Section - Checklist Items
        $this->start_controls_section(
            'checklist_content_section',
            [
                'label' => __('Checklist Items', 'relayout-design'),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'checklist_title',
            [
                'label' => __('Checklist Title', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Complete all items to unlock content', 'relayout-design'),
                'placeholder' => __('Enter checklist title', 'relayout-design'),
            ]
        );

        // Add 10 checklist item controls
        for ($i = 1; $i <= 10; $i++) {
            $this->add_control(
                "checklist_item_$i",
                [
                    'label' => sprintf(__('Checklist Item %d', 'relayout-design'), $i),
                    'type' => \Elementor\Controls_Manager::TEXT,
                    'default' => $i <= 5 ? sprintf(__('Checklist item %d', 'relayout-design'), $i) : '',
                    'placeholder' => sprintf(__('Enter checklist item %d', 'relayout-design'), $i),
                ]
            );
        }

        $this->end_controls_section();

        // Style Section - Title
        $this->start_controls_section(
            'title_style_section',
            [
                'label' => __('Title Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => __('Title Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .checklist-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'label' => __('Title Typography', 'relayout-design'),
                'selector' => '{{WRAPPER}} .checklist-title',
            ]
        );

        $this->add_responsive_control(
            'title_margin',
            [
                'label' => __('Title Margin', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .checklist-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Checklist Items
        $this->start_controls_section(
            'checklist_style_section',
            [
                'label' => __('Checklist Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'checklist_item_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .checklist-item label' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'checklist_typography',
                'label' => __('Typography', 'relayout-design'),
                'selector' => '{{WRAPPER}} .checklist-item label',
            ]
        );

        $this->add_responsive_control(
            'checklist_item_spacing',
            [
                'label' => __('Item Spacing', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 10,
                ],
                'selectors' => [
                    '{{WRAPPER}} .checklist-item' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'checkbox_size',
            [
                'label' => __('Checkbox Size', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 12,
                        'max' => 30,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 18,
                ],
                'selectors' => [
                    '{{WRAPPER}} .checklist-item input[type="checkbox"]' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'checkbox_color',
            [
                'label' => __('Checkbox Checked Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .checklist-item input[type="checkbox"]:checked' => 'background-color: {{VALUE}} !important; border-color: {{VALUE}} !important;',
                ],
            ]
        );

        $this->add_control(
            'checkbox_background',
            [
                'label' => __('Checkbox Background', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .checklist-item input[type="checkbox"]' => 'background-color: {{VALUE}} !important;',
                ],
            ]
        );

        $this->add_control(
            'checkbox_border_color',
            [
                'label' => __('Checkbox Border Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .checklist-item input[type="checkbox"]' => 'border-color: {{VALUE}} !important;',
                ],
            ]
        );

        $this->add_control(
            'checkbox_border_width',
            [
                'label' => __('Checkbox Border Width', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 5,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 1,
                ],
                'selectors' => [
                    '{{WRAPPER}} .checklist-item input[type="checkbox"]' => 'border-width: {{SIZE}}{{UNIT}} !important; border-style: solid !important;',
                ],
            ]
        );

        $this->add_control(
            'checkbox_border_radius',
            [
                'label' => __('Checkbox Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 20,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 50,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .checklist-item input[type="checkbox"]' => 'border-radius: {{SIZE}}{{UNIT}} !important;',
                ],
            ]
        );

        // Checkbox Hover States
        $this->add_control(
            'checkbox_hover_heading',
            [
                'label' => __('Hover States', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'checkbox_hover_background',
            [
                'label' => __('Hover Background', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .checklist-item input[type="checkbox"]:hover' => 'background-color: {{VALUE}} !important;',
                ],
            ]
        );

        $this->add_control(
            'checkbox_hover_border_color',
            [
                'label' => __('Hover Border Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .checklist-item input[type="checkbox"]:hover' => 'border-color: {{VALUE}} !important;',
                ],
            ]
        );

        $this->add_control(
            'checkbox_hover_transform',
            [
                'label' => __('Hover Scale', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0.8,
                        'max' => 1.2,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'size' => 1.05,
                ],
                'selectors' => [
                    '{{WRAPPER}} .checklist-item input[type="checkbox"]:hover' => 'transform: scale({{SIZE}});',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Alignment
        $this->start_controls_section(
            'alignment_style_section',
            [
                'label' => __('Alignment', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'checklist_alignment',
            [
                'label' => __('Alignment', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => __('Left', 'relayout-design'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'relayout-design'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => __('Right', 'relayout-design'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'left',
                'selectors' => [
                    '{{WRAPPER}} .checklist-manager-wrapper' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        // Check if premium features are available
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            $upgrade_prompt = relayout_designs_show_upgrade_prompt('Checklist Manager Widget');
            if ($upgrade_prompt) {
                echo $upgrade_prompt;
                return;
            }
        }
        
        $settings = $this->get_settings_for_display();
        $widget_id = $this->get_id();
        
        // Collect non-empty checklist items
        $checklist_items = [];
        for ($i = 1; $i <= 10; $i++) {
            $item_text = $settings["checklist_item_$i"];
            if (!empty($item_text)) {
                $checklist_items[] = $item_text;
            }
        }
        
        if (empty($checklist_items)) {
            return;
        }
        ?>
        <div class="checklist-manager-wrapper" data-widget-id="<?php echo esc_attr($widget_id); ?>">
            <?php if (!empty($settings['checklist_title'])): ?>
                <h3 class="checklist-title"><?php echo esc_html($settings['checklist_title']); ?></h3>
            <?php endif; ?>
            
            <div class="checklist-container">
                <?php foreach ($checklist_items as $index => $item): ?>
                    <div class="checklist-item">
                        <input type="checkbox" 
                               id="checklist-<?php echo esc_attr($widget_id); ?>-<?php echo esc_attr($index); ?>" 
                               class="checklist-checkbox" 
                               data-widget="<?php echo esc_attr($widget_id); ?>">
                        <label for="checklist-<?php echo esc_attr($widget_id); ?>-<?php echo esc_attr($index); ?>">
                            <?php echo esc_html($item); ?>
                        </label>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
    }
}
