/**
 * Cookie Consent Banner Widget JavaScript
 * Handles GDPR-compliant cookie consent functionality
 */

class CookieConsentWidget {
    constructor(element) {
        this.element = element;
        this.banner = element.querySelector('.cookie-consent-banner');
        this.overlay = element.querySelector('.cookie-overlay');
        this.acceptBtn = element.querySelector('.cookie-accept');
        this.rejectBtn = element.querySelector('.cookie-reject');
        this.preferencesBtn = element.querySelector('.cookie-preferences');
        this.closeBtn = element.querySelector('.cookie-close');
        this.categoryCheckboxes = element.querySelectorAll('.cookie-category input[type="checkbox"]');
        
        this.layout = element.dataset.layout || 'bar';
        this.position = element.dataset.position || 'bottom';
        this.floatingPosition = element.dataset.floatingPosition || 'bottom-right';
        this.autoShow = element.dataset.autoShow === 'true';
        this.showDelay = parseFloat(element.dataset.showDelay) || 0;
        this.enableCategories = element.dataset.enableCategories === 'true';
        
        this.cookieName = 'cookie_consent_choice';
        this.cookieExpiry = 365; // days
        this.preferencesVisible = false;
        
        this.init();
    }
    
    init() {
        if (!this.banner) {
            console.warn('Cookie consent banner not found');
            return;
        }
        
        console.log('Cookie consent widget initializing...', {
            autoShow: this.autoShow,
            hasUserChoice: this.hasUserChoice(),
            layout: this.layout,
            position: this.position
        });
        
        try {
            this.setupLayout();
            this.bindEvents();
            
            if (this.autoShow && !this.hasUserChoice()) {
                console.log('Auto-showing cookie consent banner');
                this.showBanner();
            } else {
                console.log('Not showing banner:', { autoShow: this.autoShow, hasUserChoice: this.hasUserChoice() });
            }
            
            // Show preferences if user clicks preferences button
            if (this.preferencesBtn) {
                this.preferencesBtn.addEventListener('click', () => {
                    this.showPreferences();
                });
            }
        } catch (error) {
            console.error('Cookie consent widget initialization failed:', error);
            // Fallback: show banner without animations
            if (this.banner) {
                this.banner.style.display = 'block';
                this.banner.style.opacity = '1';
            }
        }
    }
    
    setupLayout() {
        // Add layout classes
        this.element.classList.add(`layout-${this.layout}`);
        
        if (this.layout === 'bar') {
            this.element.classList.add(`position-${this.position}`);
        } else if (this.layout === 'floating') {
            this.element.classList.add(`floating-${this.floatingPosition}`);
        }
        
        // Set initial styles
        if (this.layout === 'modal') {
            this.banner.style.position = 'fixed';
            this.banner.style.top = '50%';
            this.banner.style.left = '50%';
            this.banner.style.transform = 'translate(-50%, -50%)';
            this.banner.style.zIndex = '10001';
            
            if (this.overlay) {
                this.overlay.style.position = 'fixed';
                this.overlay.style.top = '0';
                this.overlay.style.left = '0';
                this.overlay.style.width = '100%';
                this.overlay.style.height = '100%';
                this.overlay.style.backgroundColor = 'rgba(0, 0, 0, 0.5)';
                this.overlay.style.zIndex = '10000';
            }
        } else if (this.layout === 'bar') {
            this.banner.style.position = 'fixed';
            this.banner.style.left = '0';
            this.banner.style.right = '0';
            this.banner.style.zIndex = '10000';
            
            if (this.position === 'top') {
                this.banner.style.top = '0';
            } else {
                this.banner.style.bottom = '0';
            }
        } else if (this.layout === 'floating') {
            this.banner.style.position = 'fixed';
            this.banner.style.zIndex = '10000';
            this.banner.style.maxWidth = '400px';
            
            const [vertical, horizontal] = this.floatingPosition.split('-');
            this.banner.style[vertical] = '20px';
            this.banner.style[horizontal] = '20px';
        }
    }
    
    bindEvents() {
        if (this.acceptBtn) {
            this.acceptBtn.addEventListener('click', () => {
                this.acceptCookies();
            });
        }
        
        if (this.rejectBtn) {
            this.rejectBtn.addEventListener('click', () => {
                this.rejectCookies();
            });
        }
        
        if (this.closeBtn) {
            this.closeBtn.addEventListener('click', () => {
                this.hideBanner();
            });
        }
        
        if (this.overlay) {
            this.overlay.addEventListener('click', () => {
                this.hideBanner();
            });
        }
        
        // Keyboard support
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape' && this.isBannerVisible()) {
                this.hideBanner();
            }
        });
    }
    
    showBanner() {
        if (this.showDelay > 0) {
            setTimeout(() => {
                this.displayBanner();
            }, this.showDelay * 1000);
        } else {
            this.displayBanner();
        }
    }
    
    displayBanner() {
        if (this.overlay) {
            this.overlay.style.display = 'block';
            setTimeout(() => {
                this.overlay.style.opacity = '1';
            }, 10);
        }
        
        this.banner.style.display = 'block';
        
        // Animate in
        setTimeout(() => {
            this.banner.classList.add('show');
        }, 10);
        
        // Add body class for styling adjustments
        document.body.classList.add('cookie-consent-visible');
        
        // Dispatch custom event
        this.element.dispatchEvent(new CustomEvent('cookieConsentShown', {
            detail: { layout: this.layout }
        }));
    }
    
    hideBanner() {
        this.banner.classList.remove('show');
        
        if (this.overlay) {
            this.overlay.style.opacity = '0';
        }
        
        setTimeout(() => {
            this.banner.style.display = 'none';
            if (this.overlay) {
                this.overlay.style.display = 'none';
            }
            document.body.classList.remove('cookie-consent-visible');
        }, 300);
        
        // Dispatch custom event
        this.element.dispatchEvent(new CustomEvent('cookieConsentHidden', {
            detail: { layout: this.layout }
        }));
    }
    
    acceptCookies() {
        const consent = {
            necessary: true,
            analytics: true,
            marketing: true,
            timestamp: Date.now()
        };
        
        if (this.enableCategories) {
            // Get user preferences from checkboxes
            this.categoryCheckboxes.forEach(checkbox => {
                if (checkbox.name !== 'necessary') {
                    consent[checkbox.name] = checkbox.checked;
                }
            });
        }
        
        this.saveConsent(consent);
        this.hideBanner();
        
        // Dispatch custom event
        this.element.dispatchEvent(new CustomEvent('cookieConsentAccepted', {
            detail: { consent }
        }));
        
        // Trigger analytics/marketing scripts if accepted
        this.triggerScripts(consent);
    }
    
    rejectCookies() {
        const consent = {
            necessary: true,
            analytics: false,
            marketing: false,
            timestamp: Date.now()
        };
        
        this.saveConsent(consent);
        this.hideBanner();
        
        // Dispatch custom event
        this.element.dispatchEvent(new CustomEvent('cookieConsentRejected', {
            detail: { consent }
        }));
        
        // Remove non-necessary cookies
        this.removeNonNecessaryCookies();
    }
    
    showPreferences() {
        if (!this.enableCategories) return;
        
        const categories = this.element.querySelector('.cookie-categories');
        if (categories) {
            this.preferencesVisible = !this.preferencesVisible;
            categories.style.display = this.preferencesVisible ? 'block' : 'none';
            
            if (this.preferencesBtn) {
                this.preferencesBtn.textContent = this.preferencesVisible ? 'Hide Preferences' : 'Preferences';
            }
        }
    }
    
    saveConsent(consent) {
        // Save to localStorage
        try {
            localStorage.setItem(this.cookieName, JSON.stringify(consent));
        } catch (e) {
            console.warn('Could not save to localStorage:', e);
        }
        
        // Save to cookie as fallback
        const expiryDate = new Date();
        expiryDate.setTime(expiryDate.getTime() + (this.cookieExpiry * 24 * 60 * 60 * 1000));
        
        document.cookie = `${this.cookieName}=${encodeURIComponent(JSON.stringify(consent))}; expires=${expiryDate.toUTCString()}; path=/; SameSite=Lax`;
    }
    
    getConsent() {
        // Try localStorage first
        try {
            const stored = localStorage.getItem(this.cookieName);
            if (stored) {
                return JSON.parse(stored);
            }
        } catch (e) {
            console.warn('Could not read from localStorage:', e);
        }
        
        // Fallback to cookie
        const cookies = document.cookie.split(';');
        for (let cookie of cookies) {
            const [name, value] = cookie.trim().split('=');
            if (name === this.cookieName) {
                try {
                    return JSON.parse(decodeURIComponent(value));
                } catch (e) {
                    console.warn('Could not parse cookie value:', e);
                }
            }
        }
        
        return null;
    }
    
    hasUserChoice() {
        return this.getConsent() !== null;
    }
    
    isBannerVisible() {
        return this.banner && this.banner.style.display !== 'none' && this.banner.classList.contains('show');
    }
    
    triggerScripts(consent) {
        // Dispatch events for third-party scripts to listen to
        if (consent.analytics) {
            window.dispatchEvent(new CustomEvent('cookieConsentAnalytics', { detail: consent }));
        }
        
        if (consent.marketing) {
            window.dispatchEvent(new CustomEvent('cookieConsentMarketing', { detail: consent }));
        }
        
        // Example: Google Analytics
        if (consent.analytics && typeof gtag !== 'undefined') {
            gtag('consent', 'update', {
                'analytics_storage': 'granted'
            });
        }
        
        // Example: Google Ads
        if (consent.marketing && typeof gtag !== 'undefined') {
            gtag('consent', 'update', {
                'ad_storage': 'granted'
            });
        }
    }
    
    removeNonNecessaryCookies() {
        // Remove analytics and marketing cookies
        const cookiesToRemove = [
            '_ga', '_gid', '_gat', '_gtag', '_fbp', '_fbc'
        ];
        
        cookiesToRemove.forEach(cookieName => {
            document.cookie = `${cookieName}=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;`;
            document.cookie = `${cookieName}=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/; domain=.${window.location.hostname};`;
        });
    }
    
    // Public methods
    showConsentBanner() {
        this.showBanner();
    }
    
    hideConsentBanner() {
        this.hideBanner();
    }
    
    resetConsent() {
        localStorage.removeItem(this.cookieName);
        document.cookie = `${this.cookieName}=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;`;
        this.showBanner();
    }
    
    getUserConsent() {
        return this.getConsent();
    }
}

// Initialize widgets when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    initializeCookieConsentWidgets();
});

// Initialize for Elementor editor
(function($) {
    $(window).on('elementor/frontend/init', function() {
        if (typeof elementorFrontend !== 'undefined') {
            elementorFrontend.hooks.addAction('frontend/element_ready/cookie_consent_banner.default', function($scope) {
                initializeCookieConsentWidgets($scope[0]);
            });
        }
    });
})(jQuery || window.jQuery || window.$);

function initializeCookieConsentWidgets(container = document) {
    const widgets = container.querySelectorAll('.cookie-consent-widget');
    console.log('Initializing cookie consent widgets:', widgets.length);
    
    widgets.forEach((widget, index) => {
        // Avoid double initialization
        if (widget.cookieConsentInstance) {
            console.log(`Widget ${index} already initialized`);
            return;
        }
        
        console.log(`Initializing widget ${index}:`, widget);
        widget.cookieConsentInstance = new CookieConsentWidget(widget);
    });
}

// Handle dynamic content (AJAX, SPA)
if (window.MutationObserver) {
    const observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                mutation.addedNodes.forEach(function(node) {
                    if (node.nodeType === 1) { // Element node
                        const widgets = node.querySelectorAll ? node.querySelectorAll('.cookie-consent-widget') : [];
                        if (widgets.length > 0) {
                            initializeCookieConsentWidgets(node);
                        }
                        
                        if (node.classList && node.classList.contains('cookie-consent-widget')) {
                            initializeCookieConsentWidgets(node.parentElement);
                        }
                    }
                });
            }
        });
    });
    
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
}

// Global API
window.CookieConsentWidget = CookieConsentWidget;
window.initializeCookieConsentWidgets = initializeCookieConsentWidgets;

// Utility functions for developers
window.cookieConsent = {
    show: function() {
        const widgets = document.querySelectorAll('.cookie-consent-widget');
        console.log('Found cookie consent widgets:', widgets.length);
        widgets.forEach(widget => {
            if (widget.cookieConsentInstance) {
                console.log('Showing banner via API');
                widget.cookieConsentInstance.showConsentBanner();
            } else {
                console.log('Widget found but no instance:', widget);
            }
        });
    },
    
    hide: function() {
        const widgets = document.querySelectorAll('.cookie-consent-widget');
        widgets.forEach(widget => {
            if (widget.cookieConsentInstance) {
                widget.cookieConsentInstance.hideConsentBanner();
            }
        });
    },
    
    reset: function() {
        const widgets = document.querySelectorAll('.cookie-consent-widget');
        widgets.forEach(widget => {
            if (widget.cookieConsentInstance) {
                widget.cookieConsentInstance.resetConsent();
            }
        });
    },
    
    getConsent: function() {
        const widget = document.querySelector('.cookie-consent-widget');
        if (widget && widget.cookieConsentInstance) {
            return widget.cookieConsentInstance.getUserConsent();
        }
        return null;
    },
    
    debug: function() {
        console.log('=== Cookie Consent Debug Info ===');
        const widgets = document.querySelectorAll('.cookie-consent-widget');
        console.log('Widgets found:', widgets.length);
        
        widgets.forEach((widget, index) => {
            console.log(`Widget ${index}:`, {
                element: widget,
                hasInstance: !!widget.cookieConsentInstance,
                dataset: widget.dataset,
                banner: widget.querySelector('.cookie-consent-banner'),
                bannerStyle: widget.querySelector('.cookie-consent-banner')?.style.cssText
            });
            
            if (widget.cookieConsentInstance) {
                const instance = widget.cookieConsentInstance;
                console.log(`Instance ${index}:`, {
                    autoShow: instance.autoShow,
                    hasUserChoice: instance.hasUserChoice(),
                    layout: instance.layout,
                    position: instance.position,
                    bannerVisible: instance.isBannerVisible()
                });
            }
        });
        
        // Check if assets are loaded
        console.log('CSS loaded:', !!document.querySelector('link[href*="cookie-consent.css"]'));
        console.log('JS loaded:', typeof CookieConsentWidget !== 'undefined');
        
        return 'Debug info logged to console';
    },
    
    forceShow: function() {
        console.log('Force showing cookie consent banner...');
        const widgets = document.querySelectorAll('.cookie-consent-widget');
        widgets.forEach(widget => {
            const banner = widget.querySelector('.cookie-consent-banner');
            if (banner) {
                banner.style.display = 'block';
                banner.style.opacity = '1';
                banner.style.transform = 'translateY(0)';
                banner.classList.add('show');
                console.log('Banner forced to show');
            }
        });
    }
};
