<?php
/**
 * Cookie Consent Banner Widget for Elementor
 * 
 * @package Relayout_Design
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Make sure Elementor is active
if (!class_exists('\Elementor\Widget_Base')) {
    return;
}

class Cookie_Consent_Widget extends \Elementor\Widget_Base {

    public function get_name() {
        return 'cookie_consent_banner';
    }

    public function get_title() {
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            return __('Cookie Consent', 'relayout-design') . ' <span style="color: #ff6b35; font-weight: bold; font-size: 10px; background: #fff3f0; padding: 2px 6px; border-radius: 3px; margin-left: 5px;">PRO</span>';
        }
        return __('Cookie Consent', 'relayout-design');
    }

    public function get_icon() {
        return 'eicon-warning';
    }

    public function get_categories() {
        return ['relayout-designs'];
    }

    public function get_keywords() {
        return ['cookie', 'consent', 'gdpr', 'privacy', 'banner', 'notice'];
    }

    public function get_script_depends() {
        return ['jquery', 'cookie-consent-script'];
    }

    public function get_style_depends() {
        return ['cookie-consent-style'];
    }

    protected function register_controls() {
        // Content Section
        $this->start_controls_section(
            'content_section',
            [
                'label' => __('Content', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'banner_title',
            [
                'label' => __('Banner Title', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('We use cookies', 'relayout-design'),
                'placeholder' => __('Enter banner title...', 'relayout-design'),
            ]
        );

        $this->add_control(
            'banner_message',
            [
                'label' => __('Banner Message', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => __('We use cookies to ensure you get the best experience on our website. By continuing to use our site, you accept our use of cookies.', 'relayout-design'),
                'placeholder' => __('Enter your cookie notice message...', 'relayout-design'),
                'rows' => 3,
            ]
        );

        $this->add_control(
            'privacy_policy_text',
            [
                'label' => __('Privacy Policy Text', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Learn more', 'relayout-design'),
                'placeholder' => __('Privacy Policy', 'relayout-design'),
            ]
        );

        $this->add_control(
            'privacy_policy_url',
            [
                'label' => __('Privacy Policy URL', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::URL,
                'placeholder' => __('https://your-domain.com/privacy-policy', 'relayout-design'),
                'default' => [
                    'url' => '',
                    'is_external' => true,
                    'nofollow' => true,
                ],
            ]
        );

        $this->end_controls_section();

        // Buttons Section
        $this->start_controls_section(
            'buttons_section',
            [
                'label' => __('Buttons', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'accept_button_text',
            [
                'label' => __('Accept Button Text', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Accept All', 'relayout-design'),
                'placeholder' => __('Accept', 'relayout-design'),
            ]
        );

        $this->add_control(
            'reject_button_text',
            [
                'label' => __('Reject Button Text', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Reject All', 'relayout-design'),
                'placeholder' => __('Reject', 'relayout-design'),
            ]
        );

        $this->add_control(
            'preferences_button_text',
            [
                'label' => __('Preferences Button Text', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Preferences', 'relayout-design'),
                'placeholder' => __('Preferences', 'relayout-design'),
            ]
        );

        $this->add_control(
            'show_reject_button',
            [
                'label' => __('Show Reject Button', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'relayout-design'),
                'label_off' => __('Hide', 'relayout-design'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'show_preferences_button',
            [
                'label' => __('Show Preferences Button', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'relayout-design'),
                'label_off' => __('Hide', 'relayout-design'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();

        // Layout Section
        $this->start_controls_section(
            'layout_section',
            [
                'label' => __('Layout', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'layout_type',
            [
                'label' => __('Layout Type', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'bar',
                'options' => [
                    'bar' => __('Bar', 'relayout-design'),
                    'modal' => __('Modal', 'relayout-design'),
                    'floating' => __('Floating Box', 'relayout-design'),
                ],
            ]
        );

        $this->add_control(
            'position',
            [
                'label' => __('Position', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'bottom',
                'options' => [
                    'top' => __('Top', 'relayout-design'),
                    'bottom' => __('Bottom', 'relayout-design'),
                ],
                'condition' => [
                    'layout_type' => 'bar',
                ],
            ]
        );

        $this->add_control(
            'floating_position',
            [
                'label' => __('Floating Position', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'bottom-right',
                'options' => [
                    'bottom-left' => __('Bottom Left', 'relayout-design'),
                    'bottom-right' => __('Bottom Right', 'relayout-design'),
                    'top-left' => __('Top Left', 'relayout-design'),
                    'top-right' => __('Top Right', 'relayout-design'),
                ],
                'condition' => [
                    'layout_type' => 'floating',
                ],
            ]
        );

        $this->add_control(
            'auto_show',
            [
                'label' => __('Auto Show', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'relayout-design'),
                'label_off' => __('No', 'relayout-design'),
                'return_value' => 'yes',
                'default' => 'yes',
                'description' => __('Automatically show banner if user has not made a choice', 'relayout-design'),
            ]
        );

        $this->add_control(
            'show_delay',
            [
                'label' => __('Show Delay (seconds)', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 0,
                'min' => 0,
                'max' => 60,
                'step' => 0.5,
                'condition' => [
                    'auto_show' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        // Cookie Categories Section
        $this->start_controls_section(
            'cookie_categories_section',
            [
                'label' => __('Cookie Categories', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'enable_categories',
            [
                'label' => __('Enable Cookie Categories', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'relayout-design'),
                'label_off' => __('No', 'relayout-design'),
                'return_value' => 'yes',
                'default' => 'no',
                'description' => __('Allow users to choose specific cookie categories', 'relayout-design'),
            ]
        );

        $this->add_control(
            'necessary_cookies',
            [
                'label' => __('Necessary Cookies', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => __('These cookies are essential for the website to function properly.', 'relayout-design'),
                'condition' => [
                    'enable_categories' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'analytics_cookies',
            [
                'label' => __('Analytics Cookies', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => __('These cookies help us understand how visitors interact with our website.', 'relayout-design'),
                'condition' => [
                    'enable_categories' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'marketing_cookies',
            [
                'label' => __('Marketing Cookies', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => __('These cookies are used to deliver personalized advertisements.', 'relayout-design'),
                'condition' => [
                    'enable_categories' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Banner
        $this->start_controls_section(
            'banner_style_section',
            [
                'label' => __('Banner Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'banner_background',
                'label' => __('Background', 'relayout-design'),
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .cookie-consent-banner',
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                    'color' => [
                        'default' => '#1a1a1a',
                    ],
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'banner_border',
                'selector' => '{{WRAPPER}} .cookie-consent-banner',
            ]
        );

        $this->add_responsive_control(
            'banner_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .cookie-consent-banner' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'banner_box_shadow',
                'selector' => '{{WRAPPER}} .cookie-consent-banner',
            ]
        );

        $this->add_responsive_control(
            'banner_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'top' => 20,
                    'right' => 20,
                    'bottom' => 20,
                    'left' => 20,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .cookie-consent-banner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'banner_max_width',
            [
                'label' => __('Max Width', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'vw'],
                'range' => [
                    'px' => [
                        'min' => 300,
                        'max' => 1920,
                    ],
                    '%' => [
                        'min' => 10,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .cookie-consent-banner' => 'max-width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Title
        $this->start_controls_section(
            'title_style_section',
            [
                'label' => __('Title Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .cookie-title',
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => __('Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .cookie-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'title_margin',
            [
                'label' => __('Margin', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .cookie-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Message
        $this->start_controls_section(
            'message_style_section',
            [
                'label' => __('Message Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'message_typography',
                'selector' => '{{WRAPPER}} .cookie-message',
            ]
        );

        $this->add_control(
            'message_color',
            [
                'label' => __('Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#cccccc',
                'selectors' => [
                    '{{WRAPPER}} .cookie-message' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'link_color',
            [
                'label' => __('Link Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#4a9eff',
                'selectors' => [
                    '{{WRAPPER}} .privacy-link' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'link_hover_color',
            [
                'label' => __('Link Hover Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#6bb3ff',
                'selectors' => [
                    '{{WRAPPER}} .privacy-link:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Accept Button
        $this->start_controls_section(
            'accept_button_style_section',
            [
                'label' => __('Accept Button', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'accept_button_typography',
                'selector' => '{{WRAPPER}} .cookie-accept',
            ]
        );

        $this->start_controls_tabs('accept_button_tabs');

        $this->start_controls_tab(
            'accept_button_normal',
            [
                'label' => __('Normal', 'relayout-design'),
            ]
        );

        $this->add_control(
            'accept_button_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .cookie-accept' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'accept_button_background',
                'selector' => '{{WRAPPER}} .cookie-accept',
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                    'color' => [
                        'default' => '#28a745',
                    ],
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'accept_button_hover',
            [
                'label' => __('Hover', 'relayout-design'),
            ]
        );

        $this->add_control(
            'accept_button_hover_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .cookie-accept:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'accept_button_hover_background',
                'selector' => '{{WRAPPER}} .cookie-accept:hover',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'accept_button_border',
                'selector' => '{{WRAPPER}} .cookie-accept',
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'accept_button_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .cookie-accept' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'accept_button_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .cookie-accept' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Reject Button
        $this->start_controls_section(
            'reject_button_style_section',
            [
                'label' => __('Reject Button', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_reject_button' => 'yes',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'reject_button_typography',
                'selector' => '{{WRAPPER}} .cookie-reject',
            ]
        );

        $this->start_controls_tabs('reject_button_tabs');

        $this->start_controls_tab(
            'reject_button_normal',
            [
                'label' => __('Normal', 'relayout-design'),
            ]
        );

        $this->add_control(
            'reject_button_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .cookie-reject' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'reject_button_background',
                'selector' => '{{WRAPPER}} .cookie-reject',
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                    'color' => [
                        'default' => '#dc3545',
                    ],
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'reject_button_hover',
            [
                'label' => __('Hover', 'relayout-design'),
            ]
        );

        $this->add_control(
            'reject_button_hover_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .cookie-reject:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'reject_button_hover_background',
                'selector' => '{{WRAPPER}} .cookie-reject:hover',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'reject_button_border',
                'selector' => '{{WRAPPER}} .cookie-reject',
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'reject_button_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .cookie-reject' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'reject_button_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .cookie-reject' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Preferences Button
        $this->start_controls_section(
            'preferences_button_style_section',
            [
                'label' => __('Preferences Button', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_preferences_button' => 'yes',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'preferences_button_typography',
                'selector' => '{{WRAPPER}} .cookie-preferences',
            ]
        );

        $this->start_controls_tabs('preferences_button_tabs');

        $this->start_controls_tab(
            'preferences_button_normal',
            [
                'label' => __('Normal', 'relayout-design'),
            ]
        );

        $this->add_control(
            'preferences_button_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#1a1a1a',
                'selectors' => [
                    '{{WRAPPER}} .cookie-preferences' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'preferences_button_background',
                'selector' => '{{WRAPPER}} .cookie-preferences',
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                    'color' => [
                        'default' => '#f8f9fa',
                    ],
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'preferences_button_hover',
            [
                'label' => __('Hover', 'relayout-design'),
            ]
        );

        $this->add_control(
            'preferences_button_hover_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .cookie-preferences:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'preferences_button_hover_background',
                'selector' => '{{WRAPPER}} .cookie-preferences:hover',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'preferences_button_border',
                'selector' => '{{WRAPPER}} .cookie-preferences',
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'preferences_button_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .cookie-preferences' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'preferences_button_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .cookie-preferences' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Button Spacing
        $this->start_controls_section(
            'button_spacing_section',
            [
                'label' => __('Button Spacing', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'buttons_gap',
            [
                'label' => __('Gap Between Buttons', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                ],
                'default' => [
                    'size' => 10,
                ],
                'selectors' => [
                    '{{WRAPPER}} .cookie-buttons' => 'gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'buttons_alignment',
            [
                'label' => __('Buttons Alignment', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'flex-start' => [
                        'title' => __('Left', 'relayout-design'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'relayout-design'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'flex-end' => [
                        'title' => __('Right', 'relayout-design'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .cookie-buttons' => 'justify-content: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Cookie Categories
        $this->start_controls_section(
            'categories_style_section',
            [
                'label' => __('Cookie Categories', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'enable_categories' => 'yes',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'categories_background',
                'label' => __('Background', 'relayout-design'),
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .cookie-categories',
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                    'color' => [
                        'default' => '#f8f9fa',
                    ],
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'categories_border',
                'selector' => '{{WRAPPER}} .cookie-categories',
                'fields_options' => [
                    'border' => [
                        'default' => 'solid',
                    ],
                    'width' => [
                        'default' => [
                            'top' => 1,
                            'right' => 1,
                            'bottom' => 1,
                            'left' => 1,
                            'unit' => 'px',
                        ],
                    ],
                    'color' => [
                        'default' => '#e9ecef',
                    ],
                ],
            ]
        );

        $this->add_responsive_control(
            'categories_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 6,
                    'right' => 6,
                    'bottom' => 6,
                    'left' => 6,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .cookie-categories' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'categories_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'top' => 15,
                    'right' => 15,
                    'bottom' => 15,
                    'left' => 15,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .cookie-categories' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'categories_margin',
            [
                'label' => __('Margin', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'top' => 15,
                    'right' => 0,
                    'bottom' => 15,
                    'left' => 0,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .cookie-categories' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'category_spacing',
            [
                'label' => __('Category Item Spacing', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 30,
                    ],
                ],
                'default' => [
                    'size' => 12,
                ],
                'selectors' => [
                    '{{WRAPPER}} .cookie-category' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .cookie-category:last-child' => 'margin-bottom: 0;',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Category Labels
        $this->start_controls_section(
            'category_labels_style_section',
            [
                'label' => __('Category Labels', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'enable_categories' => 'yes',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'category_label_typography',
                'selector' => '{{WRAPPER}} .cookie-category label',
            ]
        );

        $this->add_control(
            'category_label_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}} .cookie-category label' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'category_label_spacing',
            [
                'label' => __('Label Spacing', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 20,
                    ],
                ],
                'default' => [
                    'size' => 8,
                ],
                'selectors' => [
                    '{{WRAPPER}} .cookie-category label' => 'gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Category Descriptions
        $this->start_controls_section(
            'category_descriptions_style_section',
            [
                'label' => __('Category Descriptions', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'enable_categories' => 'yes',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'category_description_typography',
                'selector' => '{{WRAPPER}} .cookie-category p',
            ]
        );

        $this->add_control(
            'category_description_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#666666',
                'selectors' => [
                    '{{WRAPPER}} .cookie-category p' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'category_description_spacing',
            [
                'label' => __('Description Spacing', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'default' => [
                    'top' => 5,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 24,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .cookie-category p' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Checkboxes
        $this->start_controls_section(
            'checkboxes_style_section',
            [
                'label' => __('Checkboxes', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'enable_categories' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'checkbox_size',
            [
                'label' => __('Checkbox Size', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 12,
                        'max' => 24,
                    ],
                ],
                'default' => [
                    'size' => 16,
                ],
                'selectors' => [
                    '{{WRAPPER}} .cookie-category input[type="checkbox"]' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'checkbox_color',
            [
                'label' => __('Checkbox Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#007cba',
                'selectors' => [
                    '{{WRAPPER}} .cookie-category input[type="checkbox"]:checked' => 'background-color: {{VALUE}}; border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'checkbox_border_color',
            [
                'label' => __('Border Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ddd',
                'selectors' => [
                    '{{WRAPPER}} .cookie-category input[type="checkbox"]' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'checkbox_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 10,
                    ],
                ],
                'default' => [
                    'size' => 3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .cookie-category input[type="checkbox"]' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        // Check if premium features are available
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            $upgrade_prompt = relayout_designs_show_upgrade_prompt('Cookie Consent Widget');
            if ($upgrade_prompt) {
                echo $upgrade_prompt;
                return;
            }
        }
        
        $settings = $this->get_settings_for_display();
        
        $layout_type = $settings['layout_type'];
        $position = isset($settings['position']) ? $settings['position'] : 'bottom';
        $floating_position = isset($settings['floating_position']) ? $settings['floating_position'] : 'bottom-right';
        $auto_show = $settings['auto_show'] === 'yes';
        $show_delay = $settings['show_delay'];
        $enable_categories = $settings['enable_categories'] === 'yes';
        
        $this->add_render_attribute('wrapper', 'class', 'cookie-consent-widget');
        $this->add_render_attribute('wrapper', 'data-layout', $layout_type);
        $this->add_render_attribute('wrapper', 'data-position', $position);
        $this->add_render_attribute('wrapper', 'data-floating-position', $floating_position);
        $this->add_render_attribute('wrapper', 'data-auto-show', $auto_show ? 'true' : 'false');
        $this->add_render_attribute('wrapper', 'data-show-delay', $show_delay);
        $this->add_render_attribute('wrapper', 'data-enable-categories', $enable_categories ? 'true' : 'false');
        
        $privacy_url = $settings['privacy_policy_url']['url'];
        $privacy_target = $settings['privacy_policy_url']['is_external'] ? '_blank' : '_self';
        $privacy_nofollow = $settings['privacy_policy_url']['nofollow'] ? 'nofollow' : '';
        
        ?>
        <div <?php echo $this->get_render_attribute_string('wrapper'); ?>>
            <div class="cookie-consent-banner" style="display: none;">
                <div class="cookie-content">
                    <?php if (!empty($settings['banner_title'])) : ?>
                        <h3 class="cookie-title"><?php echo esc_html($settings['banner_title']); ?></h3>
                    <?php endif; ?>
                    
                    <p class="cookie-message">
                        <?php echo esc_html($settings['banner_message']); ?>
                        <?php if (!empty($privacy_url) && !empty($settings['privacy_policy_text'])) : ?>
                            <a href="<?php echo esc_url($privacy_url); ?>" 
                               target="<?php echo esc_attr($privacy_target); ?>" 
                               <?php echo $privacy_nofollow ? 'rel="nofollow"' : ''; ?>
                               class="privacy-link">
                                <?php echo esc_html($settings['privacy_policy_text']); ?>
                            </a>
                        <?php endif; ?>
                    </p>
                    
                    <?php if ($enable_categories) : ?>
                        <div class="cookie-categories">
                            <div class="cookie-category">
                                <label>
                                    <input type="checkbox" name="necessary" checked disabled>
                                    <span><?php _e('Necessary', 'relayout-design'); ?></span>
                                </label>
                                <p><?php echo esc_html($settings['necessary_cookies']); ?></p>
                            </div>
                            <div class="cookie-category">
                                <label>
                                    <input type="checkbox" name="analytics">
                                    <span><?php _e('Analytics', 'relayout-design'); ?></span>
                                </label>
                                <p><?php echo esc_html($settings['analytics_cookies']); ?></p>
                            </div>
                            <div class="cookie-category">
                                <label>
                                    <input type="checkbox" name="marketing">
                                    <span><?php _e('Marketing', 'relayout-design'); ?></span>
                                </label>
                                <p><?php echo esc_html($settings['marketing_cookies']); ?></p>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
                
                <div class="cookie-buttons">
                    <button class="cookie-accept" type="button">
                        <?php echo esc_html($settings['accept_button_text']); ?>
                    </button>
                    
                    <?php if ($settings['show_reject_button'] === 'yes') : ?>
                        <button class="cookie-reject" type="button">
                            <?php echo esc_html($settings['reject_button_text']); ?>
                        </button>
                    <?php endif; ?>
                    
                    <?php if ($settings['show_preferences_button'] === 'yes') : ?>
                        <button class="cookie-preferences" type="button">
                            <?php echo esc_html($settings['preferences_button_text']); ?>
                        </button>
                    <?php endif; ?>
                </div>
                
                <?php if ($layout_type === 'modal' || $layout_type === 'floating') : ?>
                    <button class="cookie-close" type="button" aria-label="<?php _e('Close', 'relayout-design'); ?>">
                        <span>&times;</span>
                    </button>
                <?php endif; ?>
            </div>
            
            <?php if ($layout_type === 'modal') : ?>
                <div class="cookie-overlay" style="display: none;"></div>
            <?php endif; ?>
        </div>
        <?php
    }

    protected function content_template() {
        ?>
        <#
        var layoutType = settings.layout_type || 'bar';
        var position = settings.position || 'bottom';
        var floatingPosition = settings.floating_position || 'bottom-right';
        var enableCategories = settings.enable_categories === 'yes';
        var privacyUrl = settings.privacy_policy_url ? settings.privacy_policy_url.url : '';
        var privacyTarget = settings.privacy_policy_url && settings.privacy_policy_url.is_external ? '_blank' : '_self';
        #>
        
        <div class="cookie-consent-widget" data-layout="{{ layoutType }}" data-position="{{ position }}" data-floating-position="{{ floatingPosition }}" data-auto-show="true" data-show-delay="0" data-enable-categories="{{ enableCategories ? 'true' : 'false' }}">
            <div class="cookie-consent-banner" style="display: block; opacity: 1;">
                <div class="cookie-content">
                    <# if (settings.banner_title) { #>
                        <h3 class="cookie-title">{{{ settings.banner_title }}}</h3>
                    <# } #>
                    
                    <p class="cookie-message">
                        {{{ settings.banner_message }}}
                        <# if (privacyUrl && settings.privacy_policy_text) { #>
                            <a href="{{ privacyUrl }}" target="{{ privacyTarget }}" class="privacy-link">
                                {{{ settings.privacy_policy_text }}}
                            </a>
                        <# } #>
                    </p>
                    
                    <# if (enableCategories) { #>
                        <div class="cookie-categories" style="display: block;">
                            <div class="cookie-category">
                                <label>
                                    <input type="checkbox" name="necessary" checked disabled>
                                    <span><?php _e('Necessary', 'relayout-design'); ?></span>
                                </label>
                                <p>{{{ settings.necessary_cookies }}}</p>
                            </div>
                            <div class="cookie-category">
                                <label>
                                    <input type="checkbox" name="analytics">
                                    <span><?php _e('Analytics', 'relayout-design'); ?></span>
                                </label>
                                <p>{{{ settings.analytics_cookies }}}</p>
                            </div>
                            <div class="cookie-category">
                                <label>
                                    <input type="checkbox" name="marketing">
                                    <span><?php _e('Marketing', 'relayout-design'); ?></span>
                                </label>
                                <p>{{{ settings.marketing_cookies }}}</p>
                            </div>
                        </div>
                    <# } #>
                </div>
                
                <div class="cookie-buttons">
                    <button class="cookie-accept" type="button">
                        {{{ settings.accept_button_text }}}
                    </button>
                    
                    <# if (settings.show_reject_button === 'yes') { #>
                        <button class="cookie-reject" type="button">
                            {{{ settings.reject_button_text }}}
                        </button>
                    <# } #>
                    
                    <# if (settings.show_preferences_button === 'yes') { #>
                        <button class="cookie-preferences" type="button">
                            {{{ settings.preferences_button_text }}}
                        </button>
                    <# } #>
                </div>
                
                <# if (layoutType === 'modal' || layoutType === 'floating') { #>
                    <button class="cookie-close" type="button" aria-label="<?php _e('Close', 'relayout-design'); ?>">
                        <span>&times;</span>
                    </button>
                <# } #>
            </div>
            
            <# if (layoutType === 'modal') { #>
                <div class="cookie-overlay" style="display: none;"></div>
            <# } #>
        </div>
        <?php
    }
}
