/**
 * Copy to Clipboard Widget JavaScript
 * Handles clipboard functionality and tooltip animations
 */

class CopyToClipboardWidget {
    constructor(element) {
        this.element = element;
        this.button = element.querySelector('.copy-button');
        this.tooltip = element.querySelector('.copy-tooltip');
        this.textToCopy = element.dataset.text;
        this.successMessage = element.dataset.successMessage;
        
        this.init();
    }
    
    init() {
        if (!this.button) return;
        
        this.button.addEventListener('click', (e) => {
            e.preventDefault();
            this.copyToClipboard();
        });
        
        // Add keyboard support
        this.button.addEventListener('keydown', (e) => {
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                this.copyToClipboard();
            }
        });
    }
    
    async copyToClipboard() {
        try {
            // Modern clipboard API
            if (navigator.clipboard && window.isSecureContext) {
                await navigator.clipboard.writeText(this.textToCopy);
                this.showSuccess();
            } else {
                // Fallback for older browsers
                this.fallbackCopyToClipboard();
            }
        } catch (error) {
            console.warn('Copy to clipboard failed:', error);
            this.fallbackCopyToClipboard();
        }
    }
    
    fallbackCopyToClipboard() {
        try {
            // Create temporary textarea
            const textArea = document.createElement('textarea');
            textArea.value = this.textToCopy;
            textArea.style.position = 'fixed';
            textArea.style.left = '-9999px';
            textArea.style.top = '-9999px';
            document.body.appendChild(textArea);
            
            // Select and copy
            textArea.focus();
            textArea.select();
            
            const successful = document.execCommand('copy');
            document.body.removeChild(textArea);
            
            if (successful) {
                this.showSuccess();
            } else {
                this.showError();
            }
        } catch (error) {
            console.error('Fallback copy failed:', error);
            this.showError();
        }
    }
    
    showSuccess() {
        // Update button state
        this.button.classList.add('copied');
        
        // Show tooltip
        if (this.tooltip) {
            this.tooltip.textContent = this.successMessage;
            this.tooltip.classList.add('show');
            
            // Hide tooltip after delay
            setTimeout(() => {
                this.tooltip.classList.remove('show');
            }, 2000);
        }
        
        // Reset button state
        setTimeout(() => {
            this.button.classList.remove('copied');
        }, 2000);
        
        // Dispatch custom event
        this.element.dispatchEvent(new CustomEvent('copySuccess', {
            detail: {
                text: this.textToCopy,
                message: this.successMessage
            }
        }));
    }
    
    showError() {
        // Update button state
        this.button.classList.add('error');
        
        // Show error tooltip
        if (this.tooltip) {
            this.tooltip.textContent = 'Copy failed';
            this.tooltip.classList.add('show', 'error');
            
            // Hide tooltip after delay
            setTimeout(() => {
                this.tooltip.classList.remove('show', 'error');
            }, 2000);
        }
        
        // Reset button state
        setTimeout(() => {
            this.button.classList.remove('error');
        }, 2000);
        
        // Dispatch custom event
        this.element.dispatchEvent(new CustomEvent('copyError', {
            detail: {
                text: this.textToCopy
            }
        }));
    }
}

// Initialize widgets when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    initializeCopyToClipboardWidgets();
});

// Initialize for Elementor editor
jQuery(window).on('elementor/frontend/init', function() {
    elementorFrontend.hooks.addAction('frontend/element_ready/copy_to_clipboard.default', function($scope) {
        initializeCopyToClipboardWidgets($scope[0]);
    });
});

function initializeCopyToClipboardWidgets(container = document) {
    const widgets = container.querySelectorAll('.copy-to-clipboard-widget');
    
    widgets.forEach(widget => {
        // Avoid double initialization
        if (widget.copyToClipboardInstance) {
            return;
        }
        
        widget.copyToClipboardInstance = new CopyToClipboardWidget(widget);
    });
}

// Handle dynamic content (AJAX, SPA)
if (window.MutationObserver) {
    const observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                mutation.addedNodes.forEach(function(node) {
                    if (node.nodeType === 1) { // Element node
                        // Check if the added node contains copy widgets
                        const widgets = node.querySelectorAll ? node.querySelectorAll('.copy-to-clipboard-widget') : [];
                        if (widgets.length > 0) {
                            initializeCopyToClipboardWidgets(node);
                        }
                        
                        // Check if the added node itself is a copy widget
                        if (node.classList && node.classList.contains('copy-to-clipboard-widget')) {
                            initializeCopyToClipboardWidgets(node.parentElement);
                        }
                    }
                });
            }
        });
    });
    
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
}

// Export for external use
window.CopyToClipboardWidget = CopyToClipboardWidget;
window.initializeCopyToClipboardWidgets = initializeCopyToClipboardWidgets;
