<?php
/**
 * Copy to Clipboard Widget for Elementor
 * 
 * @package Relayout_Design
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Make sure Elementor is active
if (!class_exists('\Elementor\Widget_Base')) {
    return;
}

class Copy_To_Clipboard_Widget extends \Elementor\Widget_Base {

    public function get_name() {
        return 'copy_to_clipboard';
    }

    public function get_title() {
        return __('Copy to Clipboard', 'relayout-design');
    }

    public function get_icon() {
        return 'eicon-copy';
    }

    public function get_categories() {
        return ['relayout-designs'];
    }

    public function get_keywords() {
        return ['copy', 'clipboard', 'coupon', 'code', 'link', 'text'];
    }

    public function get_script_depends() {
        return ['copy-to-clipboard-widget-script'];
    }

    public function get_style_depends() {
        return ['copy-to-clipboard-widget-style'];
    }

    protected function register_controls() {
        // Content Section
        $this->start_controls_section(
            'content_section',
            [
                'label' => __('Content', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'text_to_copy',
            [
                'label' => __('Text to Copy', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('SAVE20', 'relayout-design'),
                'placeholder' => __('Enter text to copy...', 'relayout-design'),
                'description' => __('This text will be copied to clipboard when button is clicked.', 'relayout-design'),
            ]
        );

        $this->add_control(
            'display_text',
            [
                'label' => __('Display Text', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => '',
                'placeholder' => __('Leave empty to use same as copy text', 'relayout-design'),
                'description' => __('Optional: Different text to display (useful for long URLs).', 'relayout-design'),
            ]
        );

        $this->add_control(
            'button_text',
            [
                'label' => __('Button Text', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Copy', 'relayout-design'),
                'placeholder' => __('Copy', 'relayout-design'),
            ]
        );

        $this->add_control(
            'button_icon',
            [
                'label' => __('Button Icon', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-copy',
                    'library' => 'fa-solid',
                ],
            ]
        );

        $this->add_control(
            'icon_position',
            [
                'label' => __('Icon Position', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'before',
                'options' => [
                    'before' => __('Before Text', 'relayout-design'),
                    'after' => __('After Text', 'relayout-design'),
                    'only' => __('Icon Only', 'relayout-design'),
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label' => __('Icon Size', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 50,
                        'step' => 1,
                    ],
                    'em' => [
                        'min' => 0.5,
                        'max' => 3,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 16,
                ],
                'selectors' => [
                    '{{WRAPPER}} .copy-button i' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .copy-button svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_spacing',
            [
                'label' => __('Icon Spacing', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 20,
                        'step' => 1,
                    ],
                    'em' => [
                        'min' => 0,
                        'max' => 2,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 6,
                ],
                'selectors' => [
                    '{{WRAPPER}} .copy-button.icon-before .button-icon' => 'margin-right: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .copy-button.icon-after .button-icon' => 'margin-left: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'icon_position!' => 'only',
                ],
            ]
        );

        $this->add_control(
            'success_message',
            [
                'label' => __('Success Message', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Copied!', 'relayout-design'),
                'placeholder' => __('Copied!', 'relayout-design'),
            ]
        );

        $this->add_control(
            'layout',
            [
                'label' => __('Layout', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'inline',
                'options' => [
                    'inline' => __('Inline', 'relayout-design'),
                    'stacked' => __('Stacked', 'relayout-design'),
                ],
            ]
        );

        $this->add_responsive_control(
            'alignment',
            [
                'label' => __('Alignment', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => __('Left', 'relayout-design'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'relayout-design'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => __('Right', 'relayout-design'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'left',
                'selectors' => [
                    '{{WRAPPER}} .copy-to-clipboard-widget' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Text Style Section
        $this->start_controls_section(
            'text_style_section',
            [
                'label' => __('Text Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'text_typography',
                'label' => __('Typography', 'relayout-design'),
                'selector' => '{{WRAPPER}} .copy-text',
            ]
        );

        $this->add_control(
            'text_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}} .copy-text' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'text_background_color',
            [
                'label' => __('Background Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#f8f9fa',
                'selectors' => [
                    '{{WRAPPER}} .copy-text' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'text_border',
                'label' => __('Border', 'relayout-design'),
                'selector' => '{{WRAPPER}} .copy-text',
            ]
        );

        $this->add_responsive_control(
            'text_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 4,
                    'right' => 4,
                    'bottom' => 4,
                    'left' => 4,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .copy-text' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'text_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'top' => 12,
                    'right' => 16,
                    'bottom' => 12,
                    'left' => 16,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .copy-text' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Button Style Section
        $this->start_controls_section(
            'button_style_section',
            [
                'label' => __('Button Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'button_typography',
                'label' => __('Typography', 'relayout-design'),
                'selector' => '{{WRAPPER}} .copy-button',
            ]
        );

        $this->start_controls_tabs('button_style_tabs');

        $this->start_controls_tab(
            'button_normal_tab',
            [
                'label' => __('Normal', 'relayout-design'),
            ]
        );

        $this->add_control(
            'button_text_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .copy-button' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'button_background',
                'label' => __('Background', 'relayout-design'),
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .copy-button',
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                    'color' => [
                        'default' => '#007cba',
                    ],
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_hover_tab',
            [
                'label' => __('Hover', 'relayout-design'),
            ]
        );

        $this->add_control(
            'button_hover_text_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .copy-button:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'button_hover_background',
                'label' => __('Background', 'relayout-design'),
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .copy-button:hover',
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                    'color' => [
                        'default' => '#005a87',
                    ],
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'button_border',
                'label' => __('Border', 'relayout-design'),
                'selector' => '{{WRAPPER}} .copy-button',
            ]
        );

        $this->add_responsive_control(
            'button_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 4,
                    'right' => 4,
                    'bottom' => 4,
                    'left' => 4,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .copy-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'button_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'top' => 12,
                    'right' => 20,
                    'bottom' => 12,
                    'left' => 20,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .copy-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'button_margin',
            [
                'label' => __('Margin', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 8,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .copy-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'layout' => 'inline',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'button_box_shadow',
                'label' => __('Box Shadow', 'relayout-design'),
                'selector' => '{{WRAPPER}} .copy-button',
            ]
        );

        $this->end_controls_section();

        // Icon Style Section
        $this->start_controls_section(
            'icon_style_section',
            [
                'label' => __('Icon Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'button_icon[value]!' => '',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label' => __('Icon Size', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 50,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 16,
                ],
                'selectors' => [
                    '{{WRAPPER}} .copy-button .button-icon' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .copy-button .button-icon svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs('icon_style_tabs');

        $this->start_controls_tab(
            'icon_normal_tab',
            [
                'label' => __('Normal', 'relayout-design'),
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label' => __('Icon Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .copy-button .button-icon' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .copy-button .button-icon svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'icon_hover_tab',
            [
                'label' => __('Hover', 'relayout-design'),
            ]
        );

        $this->add_control(
            'icon_hover_color',
            [
                'label' => __('Icon Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .copy-button:hover .button-icon' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .copy-button:hover .button-icon svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_responsive_control(
            'icon_spacing',
            [
                'label' => __('Icon Spacing', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 30,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 6,
                ],
                'selectors' => [
                    '{{WRAPPER}} .copy-button.icon-before .button-icon' => 'margin-right: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .copy-button.icon-after .button-icon' => 'margin-left: {{SIZE}}{{UNIT}};',
                ],
                'separator' => 'before',
            ]
        );

        $this->end_controls_section();

        // Tooltip Style Section
        $this->start_controls_section(
            'tooltip_style_section',
            [
                'label' => __('Tooltip Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'tooltip_typography',
                'label' => __('Typography', 'relayout-design'),
                'selector' => '{{WRAPPER}} .copy-tooltip',
            ]
        );

        $this->add_control(
            'tooltip_text_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .copy-tooltip' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'tooltip_background_color',
            [
                'label' => __('Background Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#28a745',
                'selectors' => [
                    '{{WRAPPER}} .copy-tooltip' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .copy-tooltip::after' => 'border-top-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'tooltip_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 4,
                    'right' => 4,
                    'bottom' => 4,
                    'left' => 4,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .copy-tooltip' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function content_template() {
        ?>
        <#
        var textToCopy = settings.text_to_copy || '';
        var displayText = settings.display_text || textToCopy;
        var buttonText = settings.button_text || 'Copy';
        var successMessage = settings.success_message || 'Copied!';
        var layout = settings.layout || 'inline';
        var iconPosition = settings.icon_position || 'before';
        
        var wrapperClasses = 'copy-to-clipboard-widget layout-' + layout + ' elementor-widget-copy_to_clipboard_widget';
        var buttonClasses = 'copy-button icon-' + iconPosition;
        #>
        <div class="{{ wrapperClasses }}" data-text="{{ textToCopy }}" data-success-message="{{ successMessage }}">
            <div class="copy-container">
                <span class="copy-text">{{{ displayText }}}</span>
                <button class="{{ buttonClasses }}" type="button">
                    <# if (iconPosition === 'before' && settings.button_icon && settings.button_icon.value) { #>
                        <span class="button-icon">
                            <i class="{{ settings.button_icon.value }}" aria-hidden="true"></i>
                        </span>
                    <# } #>
                    
                    <# if (iconPosition !== 'only' && buttonText) { #>
                        <span class="button-text">{{{ buttonText }}}</span>
                    <# } #>
                    
                    <# if (iconPosition === 'after' && settings.button_icon && settings.button_icon.value) { #>
                        <span class="button-icon">
                            <i class="{{ settings.button_icon.value }}" aria-hidden="true"></i>
                        </span>
                    <# } #>
                    
                    <# if (iconPosition === 'only' && settings.button_icon && settings.button_icon.value) { #>
                        <span class="button-icon">
                            <i class="{{ settings.button_icon.value }}" aria-hidden="true"></i>
                        </span>
                    <# } #>
                </button>
                <div class="copy-tooltip">{{{ successMessage }}}</div>
            </div>
        </div>
        <?php
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        
        $text_to_copy = $settings['text_to_copy'];
        $display_text = !empty($settings['display_text']) ? $settings['display_text'] : $text_to_copy;
        $button_text = $settings['button_text'];
        $success_message = $settings['success_message'];
        $layout = $settings['layout'];
        $icon_position = $settings['icon_position'];
        
        $this->add_render_attribute('wrapper', 'class', 'copy-to-clipboard-widget');
        $this->add_render_attribute('wrapper', 'class', 'layout-' . $layout);
        $this->add_render_attribute('wrapper', 'class', 'elementor-widget-' . $this->get_name());
        $this->add_render_attribute('wrapper', 'data-text', $text_to_copy);
        $this->add_render_attribute('wrapper', 'data-success-message', $success_message);
        
        $this->add_render_attribute('button', 'class', 'copy-button');
        $this->add_render_attribute('button', 'class', 'icon-' . $icon_position);
        $this->add_render_attribute('button', 'type', 'button');
        
        ?>
        <div <?php echo $this->get_render_attribute_string('wrapper'); ?>>
            <div class="copy-container">
                <span class="copy-text"><?php echo esc_html($display_text); ?></span>
                <button <?php echo $this->get_render_attribute_string('button'); ?>>
                    <?php if ($icon_position === 'before' && !empty($settings['button_icon']['value'])) : ?>
                        <span class="button-icon">
                            <?php \Elementor\Icons_Manager::render_icon($settings['button_icon'], ['aria-hidden' => 'true']); ?>
                        </span>
                    <?php endif; ?>
                    
                    <?php if ($icon_position !== 'only' && !empty($button_text)) : ?>
                        <span class="button-text"><?php echo esc_html($button_text); ?></span>
                    <?php endif; ?>
                    
                    <?php if ($icon_position === 'after' && !empty($settings['button_icon']['value'])) : ?>
                        <span class="button-icon">
                            <?php \Elementor\Icons_Manager::render_icon($settings['button_icon'], ['aria-hidden' => 'true']); ?>
                        </span>
                    <?php endif; ?>
                    
                    <?php if ($icon_position === 'only' && !empty($settings['button_icon']['value'])) : ?>
                        <span class="button-icon">
                            <?php \Elementor\Icons_Manager::render_icon($settings['button_icon'], ['aria-hidden' => 'true']); ?>
                        </span>
                    <?php endif; ?>
                </button>
                <div class="copy-tooltip"><?php echo esc_html($success_message); ?></div>
            </div>
        </div>
        <?php
    }
}
