/**
 * Countdown Timer Widget JavaScript - Production Ready
 * 
 * @package RelayoutDesigns
 * @version 1.0.0
 * @author Relayout Designs
 * 
 * Features:
 * - Fixed Date & Evergreen Countdown modes
 * - Local Storage for evergreen persistence
 * - Performance optimized DOM updates
 * - Error handling and validation
 * - Elementor editor integration
 * - Accessibility support
 * - Memory leak prevention
 * - Cross-browser compatibility
 * 
 * Browser Support: Modern browsers (ES6+)
 * Dependencies: None (Vanilla JavaScript)
 */

class CountdownTimerWidget {
    constructor(element) {
        this.element = element;
        
        try {
            this.settings = JSON.parse(element.dataset.settings || '{}');
        } catch (error) {
            this.settings = {};
        }
        
        this.timer = element.querySelector('.countdown-timer');
        this.expiredMessage = element.querySelector('.countdown-expired-message');
        this.interval = null;
        this.targetDate = null;
        this.storageKey = 'countdown_timer_' + this.generateUniqueId();
        
        // Validate required elements
        if (!this.timer) {
            return;
        }
        
        this.init();
    }

    init() {
        // Validate settings
        if (!this.validateSettings()) {
            return;
        }
        
        this.calculateTargetDate();
        this.startCountdown();
    }
    
    validateSettings() {
        // Validate countdown mode
        if (!this.settings.mode || !['fixed', 'evergreen'].includes(this.settings.mode)) {
            return false;
        }
        
        // Validate fixed date mode
        if (this.settings.mode === 'fixed' && !this.settings.fixedDate) {
            return false;
        }
        
        // Validate evergreen mode
        if (this.settings.mode === 'evergreen') {
            const days = parseInt(this.settings.evergreenDays) || 0;
            const hours = parseInt(this.settings.evergreenHours) || 0;
            const minutes = parseInt(this.settings.evergreenMinutes) || 0;
            
            if (days === 0 && hours === 0 && minutes === 0) {
                return false;
            }
        }
        
        return true;
    }

    generateUniqueId() {
        // Generate unique ID based on widget position and settings
        const rect = this.element.getBoundingClientRect();
        const settingsString = JSON.stringify(this.settings);
        return btoa(settingsString + rect.top + rect.left).replace(/[^a-zA-Z0-9]/g, '').substring(0, 16);
    }

    calculateTargetDate() {
        if (this.settings.mode === 'fixed') {
            // Fixed date mode
            this.targetDate = new Date(this.settings.fixedDate);
        } else {
            // Evergreen mode
            this.calculateEvergreenDate();
        }
    }

    calculateEvergreenDate() {
        const stored = localStorage.getItem(this.storageKey);
        let firstVisit = null;

        if (stored) {
            const data = JSON.parse(stored);
            firstVisit = new Date(data.firstVisit);
            
            // Check if timer has expired and restart is enabled
            if (data.expired && this.settings.restartOnRevisit) {
                firstVisit = new Date();
                this.saveEvergreenData(firstVisit, false);
            }
        } else {
            // First visit
            firstVisit = new Date();
            this.saveEvergreenData(firstVisit, false);
        }

        // Calculate target date from first visit
        this.targetDate = new Date(firstVisit);
        this.targetDate.setDate(this.targetDate.getDate() + this.settings.evergreenDays);
        this.targetDate.setHours(this.targetDate.getHours() + this.settings.evergreenHours);
        this.targetDate.setMinutes(this.targetDate.getMinutes() + this.settings.evergreenMinutes);
    }

    saveEvergreenData(firstVisit, expired) {
        const data = {
            firstVisit: firstVisit.toISOString(),
            expired: expired,
            targetDate: this.targetDate.toISOString()
        };
        localStorage.setItem(this.storageKey, JSON.stringify(data));
    }

    startCountdown() {
        // Initial update
        this.updateCountdown();
        
        // Update every second
        this.interval = setInterval(() => {
            this.updateCountdown();
        }, 1000);
    }

    updateCountdown() {
        const now = new Date().getTime();
        const target = this.targetDate.getTime();
        const difference = target - now;

        if (difference <= 0) {
            this.handleExpiry();
            return;
        }

        const timeLeft = this.calculateTimeLeft(difference);
        this.displayTime(timeLeft);
    }

    calculateTimeLeft(difference) {
        const days = Math.floor(difference / (1000 * 60 * 60 * 24));
        const hours = Math.floor((difference % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
        const minutes = Math.floor((difference % (1000 * 60 * 60)) / (1000 * 60));
        const seconds = Math.floor((difference % (1000 * 60)) / 1000);

        return {
            days: days,
            hours: hours,
            minutes: minutes,
            seconds: seconds
        };
    }

    displayTime(timeLeft) {
        if (this.settings.showDays) {
            this.updateTimeUnit('days', timeLeft.days);
        }
        if (this.settings.showHours) {
            this.updateTimeUnit('hours', timeLeft.hours);
        }
        if (this.settings.showMinutes) {
            this.updateTimeUnit('minutes', timeLeft.minutes);
        }
        if (this.settings.showSeconds) {
            this.updateTimeUnit('seconds', timeLeft.seconds);
        }
    }

    updateTimeUnit(unit, value) {
        const element = this.timer.querySelector(`[data-unit="${unit}"]`);
        if (element) {
            const formattedValue = value.toString().padStart(2, '0');
            if (element.textContent !== formattedValue) {
                element.textContent = formattedValue;
                this.animateUpdate(element);
            }
        }
    }

    animateUpdate(element) {
        // Add animation class
        element.classList.add('countdown-update');
        
        // Remove animation class after animation completes
        setTimeout(() => {
            element.classList.remove('countdown-update');
        }, 300);
    }

    handleExpiry() {
        // Clear interval
        if (this.interval) {
            clearInterval(this.interval);
            this.interval = null;
        }

        // Mark as expired in evergreen mode
        if (this.settings.mode === 'evergreen') {
            const stored = localStorage.getItem(this.storageKey);
            if (stored) {
                const data = JSON.parse(stored);
                data.expired = true;
                localStorage.setItem(this.storageKey, JSON.stringify(data));
            }
        }

        // Handle expiry action
        switch (this.settings.expiryAction) {
            case 'message':
                this.showExpiryMessage();
                break;
            case 'redirect':
                this.handleRedirect();
                break;
            case 'hide':
                this.hideTimer();
                break;
        }

        // Trigger custom event
        this.element.dispatchEvent(new CustomEvent('countdownExpired', {
            detail: {
                mode: this.settings.mode,
                targetDate: this.targetDate
            }
        }));
    }

    showExpiryMessage() {
        if (this.expiredMessage) {
            this.timer.classList.add('countdown-hidden');
            this.expiredMessage.classList.remove('countdown-hidden');
            this.expiredMessage.classList.add('countdown-fade-in');
        }
    }

    handleRedirect() {
        if (this.settings.redirectUrl) {
            // Add a small delay to allow any animations to complete
            setTimeout(() => {
                window.location.href = this.settings.redirectUrl;
            }, 500);
        }
    }

    hideTimer() {
        this.timer.classList.add('countdown-fade-out');
        setTimeout(() => {
            this.timer.classList.add('countdown-hidden');
        }, 300);
    }

    destroy() {
        if (this.interval) {
            clearInterval(this.interval);
            this.interval = null;
        }
    }

    // Public method to reset evergreen timer
    resetEvergreenTimer() {
        if (this.settings.mode === 'evergreen') {
            localStorage.removeItem(this.storageKey);
            this.calculateTargetDate();
            this.startCountdown();
            
            // Show timer if it was hidden
            if (this.timer) {
                this.timer.classList.remove('countdown-hidden', 'countdown-fade-out');
            }
            if (this.expiredMessage) {
                this.expiredMessage.classList.add('countdown-hidden');
            }
        }
    }

    // Public method to get remaining time
    getRemainingTime() {
        const now = new Date().getTime();
        const target = this.targetDate.getTime();
        const difference = target - now;
        
        if (difference <= 0) {
            return null;
        }
        
        return this.calculateTimeLeft(difference);
    }
}

// Initialize widgets when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    initializeCountdownTimers();
});

// Also initialize on Elementor frontend
if (typeof window.elementorFrontend !== 'undefined') {
    window.elementorFrontend.hooks.addAction('frontend/element_ready/countdown_timer.default', function($scope) {
        initializeCountdownTimers($scope[0]);
    });
}

// Also handle Elementor editor
if (typeof window.elementor !== 'undefined') {
    window.elementor.hooks.addAction('panel/open_editor/widget/countdown_timer', function(panel, model, view) {
        // Reinitialize when widget is edited
        setTimeout(function() {
            initializeCountdownTimers();
        }, 100);
    });
}

// Initialize function
function initializeCountdownTimers(container = document) {
    const widgets = container.querySelectorAll('.countdown-timer-widget');
    
    if (widgets.length === 0) {
        return;
    }
    
    widgets.forEach((widget, index) => {
        try {
            // Avoid double initialization
            if (widget.countdownInstance) {
                widget.countdownInstance.destroy();
            }
            
            // Check if widget has required data
            if (!widget.dataset.settings) {
                return;
            }
            
            widget.countdownInstance = new CountdownTimerWidget(widget);
        } catch (error) {
            // Silent error handling for production
            if (window.console && console.error) {
                console.error('Countdown Timer initialization failed:', error);
            }
        }
    });
}

// Handle SPA navigation and dynamic content
if (typeof MutationObserver !== 'undefined') {
    const observer = new MutationObserver(function(mutations) {
        let shouldReinit = false;
        
        mutations.forEach(function(mutation) {
            if (mutation.type === 'childList') {
                mutation.addedNodes.forEach(function(node) {
                    if (node.nodeType === 1) { // Element node
                        if (node.matches && node.matches('.countdown-timer-widget')) {
                            shouldReinit = true;
                        } else if (node.querySelector && node.querySelector('.countdown-timer-widget')) {
                            shouldReinit = true;
                        }
                    }
                });
            }
        });
        
        if (shouldReinit) {
            setTimeout(initializeCountdownTimers, 100);
        }
    });
    
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
}

// Utility functions for external use
window.CountdownTimerUtils = {
    // Reset all evergreen timers on the page
    resetAllEvergreenTimers: function() {
        const widgets = document.querySelectorAll('.countdown-timer-widget');
        widgets.forEach(widget => {
            if (widget.countdownInstance) {
                widget.countdownInstance.resetEvergreenTimer();
            }
        });
    },
    
    // Get remaining time for a specific timer
    getRemainingTime: function(widgetElement) {
        if (widgetElement && widgetElement.countdownInstance) {
            return widgetElement.countdownInstance.getRemainingTime();
        }
        return null;
    },
    
    // Clear all evergreen timer data from localStorage
    clearAllEvergreenData: function() {
        const keys = Object.keys(localStorage);
        keys.forEach(key => {
            if (key.startsWith('countdown_timer_')) {
                localStorage.removeItem(key);
            }
        });
    }
};

// Expose for external use
window.CountdownTimerWidget = CountdownTimerWidget;
window.initializeCountdownTimers = initializeCountdownTimers;
