<?php
/**
 * Countdown Timer Widget for Elementor
 * Supports fixed date countdown and evergreen countdown with multiple styles
 * 
 * Features:
 * - 8 Awesome Styles: Modern Block, Circle, Flip Card, Neon Glow, Progress Ring, Glass Morphism, Minimal Line, Inline
 * - Animation Effects: Pulse, Bounce, Glow, Shake
 * - Color Schemes: Gradient variations, Neon effects, Dark mode, Minimal mono
 * - Progress Indicators: Progress bar and percentage display
 * - Smart Features: Sound alerts, Milestone alerts, Urgent styling
 * - Responsive Design: Mobile-first approach with proper breakpoints
 * - Accessibility: ARIA support, keyboard navigation, high contrast mode
 * - Performance: Optimized animations, efficient DOM manipulation
 *
 * @package RelayoutDesigns
 * @version 1.0.0
 * @author Relayout Designs
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Make sure Elementor is active
if (!class_exists('\Elementor\Widget_Base')) {
    return;
}

class Countdown_Timer_Widget extends \Elementor\Widget_Base {

    public function get_name() {
        return 'countdown_timer';
    }

    public function get_title() {
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            return __('Countdown Timer', 'relayout-design') . ' <span class="relayout-pro-badge">PRO</span>';
        }
        return __('Countdown Timer', 'relayout-design');
    }

    public function get_icon() {
        return 'eicon-countdown';
    }

    public function get_categories() {
        return ['relayout-designs'];
    }

    public function get_script_depends() {
        return ['countdown-timer-widget-script'];
    }

    public function get_style_depends() {
        return ['countdown-timer-widget-style'];
    }

    protected function register_controls() {
        // Content Section - Timer Settings
        $this->start_controls_section(
            'timer_settings_section',
            [
                'label' => __('Timer Settings', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'countdown_mode',
            [
                'label' => __('Countdown Mode', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'fixed',
                'options' => [
                    'fixed' => __('Fixed Date', 'relayout-design'),
                    'evergreen' => __('Evergreen (Days from first visit)', 'relayout-design'),
                ],
            ]
        );

        $this->add_control(
            'fixed_date',
            [
                'label' => __('Target Date', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DATE_TIME,
                'default' => date('Y-m-d H:i', strtotime('+30 days')),
                'condition' => [
                    'countdown_mode' => 'fixed',
                ],
                'description' => __('Set the target date and time for countdown', 'relayout-design'),
            ]
        );

        $this->add_control(
            'evergreen_days',
            [
                'label' => __('Days from First Visit', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 7,
                'min' => 1,
                'max' => 365,
                'condition' => [
                    'countdown_mode' => 'evergreen',
                ],
                'description' => __('Number of days to countdown from user\'s first visit', 'relayout-design'),
            ]
        );

        $this->add_control(
            'evergreen_hours',
            [
                'label' => __('Additional Hours', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 0,
                'min' => 0,
                'max' => 23,
                'condition' => [
                    'countdown_mode' => 'evergreen',
                ],
            ]
        );

        $this->add_control(
            'evergreen_minutes',
            [
                'label' => __('Additional Minutes', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 0,
                'min' => 0,
                'max' => 59,
                'condition' => [
                    'countdown_mode' => 'evergreen',
                ],
            ]
        );

        $this->add_control(
            'restart_on_revisit',
            [
                'label' => __('Restart on Revisit', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'no',
                'condition' => [
                    'countdown_mode' => 'evergreen',
                ],
                'description' => __('Reset countdown timer when user revisits after expiry', 'relayout-design'),
            ]
        );

        $this->end_controls_section();

        // Content Section - Display Settings
        $this->start_controls_section(
            'display_settings_section',
            [
                'label' => __('Display Settings', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'timer_style',
            [
                'label' => __('Timer Style', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'modern-block',
                'options' => [
                    'inline' => __('Inline', 'relayout-design'),
                    'modern-block' => __('Modern Block', 'relayout-design'),
                    'circle' => __('Circle', 'relayout-design'),
                    'flip-card' => __('Flip Card', 'relayout-design'),
                    'neon-glow' => __('Neon Glow', 'relayout-design'),
                    'progress-ring' => __('Progress Ring', 'relayout-design'),
                    'glass-morphism' => __('Glass Morphism', 'relayout-design'),
                    'minimal-line' => __('Minimal Line', 'relayout-design'),
                ],
            ]
        );

        $this->add_control(
            'animation_style',
            [
                'label' => __('Animation Style', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'none',
                'options' => [
                    'none' => __('None', 'relayout-design'),
                    'pulse' => __('Pulse', 'relayout-design'),
                    'bounce' => __('Bounce', 'relayout-design'),
                    'glow' => __('Glow', 'relayout-design'),
                    'shake' => __('Shake', 'relayout-design'),
                    'flip' => __('Flip on Update', 'relayout-design'),
                ],
            ]
        );

        $this->add_control(
            'color_scheme',
            [
                'label' => __('Color Scheme', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'default',
                'options' => [
                    'default' => __('Default', 'relayout-design'),
                    'gradient-blue' => __('Gradient Blue', 'relayout-design'),
                    'gradient-purple' => __('Gradient Purple', 'relayout-design'),
                    'gradient-orange' => __('Gradient Orange', 'relayout-design'),
                    'neon-cyan' => __('Neon Cyan', 'relayout-design'),
                    'neon-pink' => __('Neon Pink', 'relayout-design'),
                    'dark-mode' => __('Dark Mode', 'relayout-design'),
                    'minimal-mono' => __('Minimal Mono', 'relayout-design'),
                ],
            ]
        );

        $this->add_control(
            'show_days',
            [
                'label' => __('Show Days', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'show_hours',
            [
                'label' => __('Show Hours', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'show_minutes',
            [
                'label' => __('Show Minutes', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'show_seconds',
            [
                'label' => __('Show Seconds', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'labels_position',
            [
                'label' => __('Labels Position', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'bottom',
                'options' => [
                    'top' => __('Top', 'relayout-design'),
                    'bottom' => __('Bottom', 'relayout-design'),
                    'inside' => __('Inside', 'relayout-design'),
                ],
                'condition' => [
                    'timer_style!' => 'inline',
                ],
            ]
        );

        $this->end_controls_section();

        // Content Section - Labels
        $this->start_controls_section(
            'labels_section',
            [
                'label' => __('Labels', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'label_days',
            [
                'label' => __('Days Label', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Days', 'relayout-design'),
                'condition' => [
                    'show_days' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'label_hours',
            [
                'label' => __('Hours Label', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Hours', 'relayout-design'),
                'condition' => [
                    'show_hours' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'label_minutes',
            [
                'label' => __('Minutes Label', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Minutes', 'relayout-design'),
                'condition' => [
                    'show_minutes' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'label_seconds',
            [
                'label' => __('Seconds Label', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Seconds', 'relayout-design'),
                'condition' => [
                    'show_seconds' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'show_progress',
            [
                'label' => __('Show Progress Bar', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'no',
                'condition' => [
                    'timer_style!' => ['progress-ring', 'minimal-line'],
                ],
            ]
        );

        $this->add_control(
            'show_percentage',
            [
                'label' => __('Show Percentage', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'no',
            ]
        );

        $this->end_controls_section();

        // Content Section - Sound & Alerts
        $this->start_controls_section(
            'alerts_section',
            [
                'label' => __('Sound & Alerts', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'enable_sound',
            [
                'label' => __('Enable Sound Alerts', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'no',
                'description' => __('Play sound alerts at milestones and completion', 'relayout-design'),
            ]
        );

        $this->add_control(
            'milestone_alerts',
            [
                'label' => __('Milestone Alerts', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => '24:00:00,1:00:00,0:10:00,0:01:00',
                'description' => __('Comma-separated time values (format: H:MM:SS or D:HH:MM:SS)', 'relayout-design'),
                'condition' => [
                    'enable_sound' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'urgent_threshold',
            [
                'label' => __('Urgent Threshold (minutes)', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 10,
                'min' => 1,
                'max' => 1440,
                'description' => __('When to show urgent styling (red/pulsing)', 'relayout-design'),
            ]
        );

        $this->end_controls_section();

        // Content Section - Expiry Actions
        $this->start_controls_section(
            'expiry_actions_section',
            [
                'label' => __('Expiry Actions', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'expiry_action',
            [
                'label' => __('Action on Expiry', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'message',
                'options' => [
                    'message' => __('Show Message', 'relayout-design'),
                    'redirect' => __('Redirect to URL', 'relayout-design'),
                    'hide' => __('Hide Timer', 'relayout-design'),
                ],
            ]
        );

        $this->add_control(
            'expiry_message',
            [
                'label' => __('Expiry Message', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::WYSIWYG,
                'default' => __('Time\'s up! The countdown has ended.', 'relayout-design'),
                'condition' => [
                    'expiry_action' => 'message',
                ],
            ]
        );

        $this->add_control(
            'redirect_url',
            [
                'label' => __('Redirect URL', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::URL,
                'placeholder' => __('https://your-link.com', 'relayout-design'),
                'condition' => [
                    'expiry_action' => 'redirect',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Timer Numbers
        $this->start_controls_section(
            'numbers_style_section',
            [
                'label' => __('Timer Numbers', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'numbers_typography',
                'label' => __('Typography', 'relayout-design'),
                'selector' => '{{WRAPPER}} .countdown-number',
            ]
        );

        $this->add_control(
            'numbers_color',
            [
                'label' => __('Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}} .countdown-number' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'numbers_background',
            [
                'label' => __('Background Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .countdown-item' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'numbers_border',
                'label' => __('Border', 'relayout-design'),
                'selector' => '{{WRAPPER}} .countdown-item',
            ]
        );

        $this->add_responsive_control(
            'numbers_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 8,
                    'right' => 8,
                    'bottom' => 8,
                    'left' => 8,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .countdown-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'numbers_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'default' => [
                    'top' => 20,
                    'right' => 15,
                    'bottom' => 20,
                    'left' => 15,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .countdown-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'numbers_box_shadow',
                'label' => __('Box Shadow', 'relayout-design'),
                'selector' => '{{WRAPPER}} .countdown-item',
            ]
        );

        $this->end_controls_section();

        // Style Section - Labels
        $this->start_controls_section(
            'labels_style_section',
            [
                'label' => __('Labels Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'labels_typography',
                'label' => __('Typography', 'relayout-design'),
                'selector' => '{{WRAPPER}} .countdown-label',
            ]
        );

        $this->add_control(
            'labels_color',
            [
                'label' => __('Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#666666',
                'selectors' => [
                    '{{WRAPPER}} .countdown-label' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'labels_spacing',
            [
                'label' => __('Spacing', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 10,
                ],
                'selectors' => [
                    '{{WRAPPER}} .labels-top .countdown-label' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .labels-bottom .countdown-label' => 'margin-top: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Container
        $this->start_controls_section(
            'container_style_section',
            [
                'label' => __('Container Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'items_spacing',
            [
                'label' => __('Items Spacing', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 15,
                ],
                'selectors' => [
                    '{{WRAPPER}} .countdown-timer.style-inline' => 'gap: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .countdown-timer.style-block' => 'gap: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .countdown-timer.style-circle' => 'gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'container_alignment',
            [
                'label' => __('Alignment', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => __('Left', 'relayout-design'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'relayout-design'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => __('Right', 'relayout-design'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .countdown-timer' => 'justify-content: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Expiry Message
        $this->start_controls_section(
            'expiry_message_style_section',
            [
                'label' => __('Expiry Message Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'expiry_action' => 'message',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'expiry_message_typography',
                'label' => __('Typography', 'relayout-design'),
                'selector' => '{{WRAPPER}} .countdown-expired-message',
            ]
        );

        $this->add_control(
            'expiry_message_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}} .countdown-expired-message' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'expiry_message_background',
            [
                'label' => __('Background Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#f8f9fa',
                'selectors' => [
                    '{{WRAPPER}} .countdown-expired-message' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'expiry_message_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'default' => [
                    'top' => 20,
                    'right' => 20,
                    'bottom' => 20,
                    'left' => 20,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .countdown-expired-message' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'expiry_message_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 8,
                    'right' => 8,
                    'bottom' => 8,
                    'left' => 8,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .countdown-expired-message' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        // Check if premium features are available
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            $upgrade_prompt = relayout_designs_show_upgrade_prompt('Countdown Timer Widget');
            if ($upgrade_prompt) {
                echo $upgrade_prompt;
                return;
            }
        }
        
        $settings = $this->get_settings_for_display();
        
        // Safety checks for required settings
        if (empty($settings)) {
            echo '<div class="countdown-error">Widget settings not found.</div>';
            return;
        }
        
        // Validate countdown mode and date
        if ($settings['countdown_mode'] === 'fixed' && empty($settings['fixed_date'])) {
            echo '<div class="countdown-error">Please set a target date for the countdown.</div>';
            return;
        }
        
        $widget_settings = [
            'mode' => $settings['countdown_mode'],
            'fixedDate' => $settings['fixed_date'] ?? '',
            'evergreenDays' => intval($settings['evergreen_days'] ?? 7),
            'evergreenHours' => intval($settings['evergreen_hours'] ?? 0),
            'evergreenMinutes' => intval($settings['evergreen_minutes'] ?? 0),
            'restartOnRevisit' => $settings['restart_on_revisit'] === 'yes',
            'expiryAction' => $settings['expiry_action'],
            'redirectUrl' => $settings['redirect_url']['url'] ?? '',
            'showDays' => $settings['show_days'] === 'yes',
            'showHours' => $settings['show_hours'] === 'yes',
            'showMinutes' => $settings['show_minutes'] === 'yes',
            'showSeconds' => $settings['show_seconds'] === 'yes',
            'showProgress' => $settings['show_progress'] === 'yes',
            'showPercentage' => $settings['show_percentage'] === 'yes',
            'enableSound' => $settings['enable_sound'] === 'yes',
            'milestoneAlerts' => $settings['milestone_alerts'] ?? '',
            'urgentThreshold' => intval($settings['urgent_threshold'] ?? 10),
            'animationStyle' => $settings['animation_style'] ?? 'none',
        ];

        $labels_position_class = 'labels-' . ($settings['labels_position'] ?? 'bottom');
        $animation_class = $settings['animation_style'] !== 'none' ? 'animation-' . $settings['animation_style'] : '';
        $color_scheme_class = $settings['color_scheme'] !== 'default' ? 'color-' . $settings['color_scheme'] : '';
        ?>
        
        <div class="countdown-timer-widget <?php echo esc_attr($animation_class); ?> <?php echo esc_attr($color_scheme_class); ?>" 
             data-settings="<?php echo esc_attr(json_encode($widget_settings)); ?>"
             role="timer" 
             aria-label="<?php echo esc_attr__('Countdown Timer', 'relayout-design'); ?>"
             aria-live="polite">
            <?php if ($settings['show_progress'] === 'yes' && !in_array($settings['timer_style'], ['progress-ring', 'minimal-line'])) : ?>
                <div class="countdown-progress-bar">
                    <div class="countdown-progress-fill" data-progress="0"></div>
                </div>
            <?php endif; ?>
            
            <?php if ($settings['show_percentage'] === 'yes') : ?>
                <div class="countdown-percentage">0%</div>
            <?php endif; ?>
            
            <div class="countdown-timer style-<?php echo esc_attr($settings['timer_style']); ?> <?php echo esc_attr($labels_position_class); ?>">
                
                <?php if ($settings['show_days'] === 'yes') : ?>
                    <div class="countdown-item countdown-days">
                        <?php if (in_array($settings['timer_style'], ['flip-card'])) : ?>
                            <div class="flip-container">
                                <div class="flip-card">
                                    <div class="flip-front">
                                        <span class="countdown-number" data-unit="days">00</span>
                                    </div>
                                    <div class="flip-back">
                                        <span class="countdown-number" data-unit="days">00</span>
                                    </div>
                                </div>
                            </div>
                        <?php elseif (in_array($settings['timer_style'], ['progress-ring'])) : ?>
                            <div class="progress-ring-container">
                                <svg class="progress-ring" width="120" height="120">
                                    <circle class="progress-ring-bg" cx="60" cy="60" r="54"></circle>
                                    <circle class="progress-ring-fill" cx="60" cy="60" r="54" data-unit="days"></circle>
                                </svg>
                                <div class="progress-ring-content">
                                    <div class="countdown-number" data-unit="days">00</div>
                                </div>
                            </div>
                        <?php else : ?>
                            <?php if ($settings['labels_position'] === 'top') : ?>
                                <div class="countdown-label"><?php echo esc_html($settings['label_days']); ?></div>
                            <?php endif; ?>
                            <div class="countdown-number" data-unit="days">00</div>
                            <?php if ($settings['labels_position'] === 'bottom') : ?>
                                <div class="countdown-label"><?php echo esc_html($settings['label_days']); ?></div>
                            <?php endif; ?>
                        <?php endif; ?>
                        <?php if (in_array($settings['timer_style'], ['flip-card', 'progress-ring'])) : ?>
                            <div class="countdown-label"><?php echo esc_html($settings['label_days']); ?></div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <?php if ($settings['show_hours'] === 'yes') : ?>
                    <div class="countdown-item countdown-hours">
                        <?php if (in_array($settings['timer_style'], ['flip-card'])) : ?>
                            <div class="flip-container">
                                <div class="flip-card">
                                    <div class="flip-front">
                                        <span class="countdown-number" data-unit="hours">00</span>
                                    </div>
                                    <div class="flip-back">
                                        <span class="countdown-number" data-unit="hours">00</span>
                                    </div>
                                </div>
                            </div>
                        <?php elseif (in_array($settings['timer_style'], ['progress-ring'])) : ?>
                            <div class="progress-ring-container">
                                <svg class="progress-ring" width="120" height="120">
                                    <circle class="progress-ring-bg" cx="60" cy="60" r="54"></circle>
                                    <circle class="progress-ring-fill" cx="60" cy="60" r="54" data-unit="hours"></circle>
                                </svg>
                                <div class="progress-ring-content">
                                    <div class="countdown-number" data-unit="hours">00</div>
                                </div>
                            </div>
                        <?php else : ?>
                            <?php if ($settings['labels_position'] === 'top') : ?>
                                <div class="countdown-label"><?php echo esc_html($settings['label_hours']); ?></div>
                            <?php endif; ?>
                            <div class="countdown-number" data-unit="hours">00</div>
                            <?php if ($settings['labels_position'] === 'bottom') : ?>
                                <div class="countdown-label"><?php echo esc_html($settings['label_hours']); ?></div>
                            <?php endif; ?>
                        <?php endif; ?>
                        <?php if (in_array($settings['timer_style'], ['flip-card', 'progress-ring'])) : ?>
                            <div class="countdown-label"><?php echo esc_html($settings['label_hours']); ?></div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <?php if ($settings['show_minutes'] === 'yes') : ?>
                    <div class="countdown-item countdown-minutes">
                        <?php if (in_array($settings['timer_style'], ['flip-card'])) : ?>
                            <div class="flip-container">
                                <div class="flip-card">
                                    <div class="flip-front">
                                        <span class="countdown-number" data-unit="minutes">00</span>
                                    </div>
                                    <div class="flip-back">
                                        <span class="countdown-number" data-unit="minutes">00</span>
                                    </div>
                                </div>
                            </div>
                        <?php elseif (in_array($settings['timer_style'], ['progress-ring'])) : ?>
                            <div class="progress-ring-container">
                                <svg class="progress-ring" width="120" height="120">
                                    <circle class="progress-ring-bg" cx="60" cy="60" r="54"></circle>
                                    <circle class="progress-ring-fill" cx="60" cy="60" r="54" data-unit="minutes"></circle>
                                </svg>
                                <div class="progress-ring-content">
                                    <div class="countdown-number" data-unit="minutes">00</div>
                                </div>
                            </div>
                        <?php else : ?>
                            <?php if ($settings['labels_position'] === 'top') : ?>
                                <div class="countdown-label"><?php echo esc_html($settings['label_minutes']); ?></div>
                            <?php endif; ?>
                            <div class="countdown-number" data-unit="minutes">00</div>
                            <?php if ($settings['labels_position'] === 'bottom') : ?>
                                <div class="countdown-label"><?php echo esc_html($settings['label_minutes']); ?></div>
                            <?php endif; ?>
                        <?php endif; ?>
                        <?php if (in_array($settings['timer_style'], ['flip-card', 'progress-ring'])) : ?>
                            <div class="countdown-label"><?php echo esc_html($settings['label_minutes']); ?></div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <?php if ($settings['show_seconds'] === 'yes') : ?>
                    <div class="countdown-item countdown-seconds">
                        <?php if (in_array($settings['timer_style'], ['flip-card'])) : ?>
                            <div class="flip-container">
                                <div class="flip-card">
                                    <div class="flip-front">
                                        <span class="countdown-number" data-unit="seconds">00</span>
                                    </div>
                                    <div class="flip-back">
                                        <span class="countdown-number" data-unit="seconds">00</span>
                                    </div>
                                </div>
                            </div>
                        <?php elseif (in_array($settings['timer_style'], ['progress-ring'])) : ?>
                            <div class="progress-ring-container">
                                <svg class="progress-ring" width="120" height="120">
                                    <circle class="progress-ring-bg" cx="60" cy="60" r="54"></circle>
                                    <circle class="progress-ring-fill" cx="60" cy="60" r="54" data-unit="seconds"></circle>
                                </svg>
                                <div class="progress-ring-content">
                                    <div class="countdown-number" data-unit="seconds">00</div>
                                </div>
                            </div>
                        <?php else : ?>
                            <?php if ($settings['labels_position'] === 'top') : ?>
                                <div class="countdown-label"><?php echo esc_html($settings['label_seconds']); ?></div>
                            <?php endif; ?>
                            <div class="countdown-number" data-unit="seconds">00</div>
                            <?php if ($settings['labels_position'] === 'bottom') : ?>
                                <div class="countdown-label"><?php echo esc_html($settings['label_seconds']); ?></div>
                            <?php endif; ?>
                        <?php endif; ?>
                        <?php if (in_array($settings['timer_style'], ['flip-card', 'progress-ring'])) : ?>
                            <div class="countdown-label"><?php echo esc_html($settings['label_seconds']); ?></div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

            </div>

            <?php if ($settings['expiry_action'] === 'message') : ?>
                <div class="countdown-expired-message countdown-hidden">
                    <?php echo wp_kses_post($settings['expiry_message']); ?>
                </div>
            <?php endif; ?>
        </div>
        
        <?php
    }

    protected function content_template() {
        ?>
        <#
        var showDays = settings.show_days === 'yes';
        var showHours = settings.show_hours === 'yes';
        var showMinutes = settings.show_minutes === 'yes';
        var showSeconds = settings.show_seconds === 'yes';
        var labelsPosition = settings.labels_position || 'bottom';
        var timerStyle = settings.timer_style || 'modern-block';
        var labelsPositionClass = 'labels-' + labelsPosition;
        #>
        
        <# 
        var animationClass = settings.animation_style !== 'none' ? 'animation-' + settings.animation_style : '';
        var colorSchemeClass = settings.color_scheme !== 'default' ? 'color-' + settings.color_scheme : '';
        #>
        
        <div class="countdown-timer-widget {{ animationClass }} {{ colorSchemeClass }}">
            <# if (settings.show_progress === 'yes' && timerStyle !== 'progress-ring' && timerStyle !== 'minimal-line') { #>
                <div class="countdown-progress-bar">
                    <div class="countdown-progress-fill" data-progress="50"></div>
                </div>
            <# } #>
            
            <# if (settings.show_percentage === 'yes') { #>
                <div class="countdown-percentage">50%</div>
            <# } #>
            
            <div class="countdown-timer style-{{ timerStyle }} {{ labelsPositionClass }}">
                
                <# if (showDays) { #>
                    <div class="countdown-item countdown-days">
                        <# if (timerStyle === 'flip-card') { #>
                            <div class="flip-container">
                                <div class="flip-card">
                                    <div class="flip-front">
                                        <span class="countdown-number" data-unit="days">15</span>
                                    </div>
                                    <div class="flip-back">
                                        <span class="countdown-number" data-unit="days">15</span>
                                    </div>
                                </div>
                            </div>
                            <div class="countdown-label">{{{ settings.label_days }}}</div>
                        <# } else if (timerStyle === 'progress-ring') { #>
                            <div class="progress-ring-container">
                                <svg class="progress-ring" width="120" height="120">
                                    <circle class="progress-ring-bg" cx="60" cy="60" r="54"></circle>
                                    <circle class="progress-ring-fill" cx="60" cy="60" r="54" data-unit="days"></circle>
                                </svg>
                                <div class="progress-ring-content">
                                    <div class="countdown-number" data-unit="days">15</div>
                                </div>
                            </div>
                            <div class="countdown-label">{{{ settings.label_days }}}</div>
                        <# } else { #>
                            <# if (labelsPosition === 'top') { #>
                                <div class="countdown-label">{{{ settings.label_days }}}</div>
                            <# } #>
                            <div class="countdown-number" data-unit="days">15</div>
                            <# if (labelsPosition === 'bottom') { #>
                                <div class="countdown-label">{{{ settings.label_days }}}</div>
                            <# } #>
                        <# } #>
                    </div>
                <# } #>

                <# if (showHours) { #>
                    <div class="countdown-item countdown-hours">
                        <# if (timerStyle === 'flip-card') { #>
                            <div class="flip-container">
                                <div class="flip-card">
                                    <div class="flip-front">
                                        <span class="countdown-number" data-unit="hours">08</span>
                                    </div>
                                    <div class="flip-back">
                                        <span class="countdown-number" data-unit="hours">08</span>
                                    </div>
                                </div>
                            </div>
                            <div class="countdown-label">{{{ settings.label_hours }}}</div>
                        <# } else if (timerStyle === 'progress-ring') { #>
                            <div class="progress-ring-container">
                                <svg class="progress-ring" width="120" height="120">
                                    <circle class="progress-ring-bg" cx="60" cy="60" r="54"></circle>
                                    <circle class="progress-ring-fill" cx="60" cy="60" r="54" data-unit="hours"></circle>
                                </svg>
                                <div class="progress-ring-content">
                                    <div class="countdown-number" data-unit="hours">08</div>
                                </div>
                            </div>
                            <div class="countdown-label">{{{ settings.label_hours }}}</div>
                        <# } else { #>
                            <# if (labelsPosition === 'top') { #>
                                <div class="countdown-label">{{{ settings.label_hours }}}</div>
                            <# } #>
                            <div class="countdown-number" data-unit="hours">08</div>
                            <# if (labelsPosition === 'bottom') { #>
                                <div class="countdown-label">{{{ settings.label_hours }}}</div>
                            <# } #>
                        <# } #>
                    </div>
                <# } #>

                <# if (showMinutes) { #>
                    <div class="countdown-item countdown-minutes">
                        <# if (timerStyle === 'flip-card') { #>
                            <div class="flip-container">
                                <div class="flip-card">
                                    <div class="flip-front">
                                        <span class="countdown-number" data-unit="minutes">42</span>
                                    </div>
                                    <div class="flip-back">
                                        <span class="countdown-number" data-unit="minutes">42</span>
                                    </div>
                                </div>
                            </div>
                            <div class="countdown-label">{{{ settings.label_minutes }}}</div>
                        <# } else if (timerStyle === 'progress-ring') { #>
                            <div class="progress-ring-container">
                                <svg class="progress-ring" width="120" height="120">
                                    <circle class="progress-ring-bg" cx="60" cy="60" r="54"></circle>
                                    <circle class="progress-ring-fill" cx="60" cy="60" r="54" data-unit="minutes"></circle>
                                </svg>
                                <div class="progress-ring-content">
                                    <div class="countdown-number" data-unit="minutes">42</div>
                                </div>
                            </div>
                            <div class="countdown-label">{{{ settings.label_minutes }}}</div>
                        <# } else { #>
                            <# if (labelsPosition === 'top') { #>
                                <div class="countdown-label">{{{ settings.label_minutes }}}</div>
                            <# } #>
                            <div class="countdown-number" data-unit="minutes">42</div>
                            <# if (labelsPosition === 'bottom') { #>
                                <div class="countdown-label">{{{ settings.label_minutes }}}</div>
                            <# } #>
                        <# } #>
                    </div>
                <# } #>

                <# if (showSeconds) { #>
                    <div class="countdown-item countdown-seconds">
                        <# if (timerStyle === 'flip-card') { #>
                            <div class="flip-container">
                                <div class="flip-card">
                                    <div class="flip-front">
                                        <span class="countdown-number" data-unit="seconds">30</span>
                                    </div>
                                    <div class="flip-back">
                                        <span class="countdown-number" data-unit="seconds">30</span>
                                    </div>
                                </div>
                            </div>
                            <div class="countdown-label">{{{ settings.label_seconds }}}</div>
                        <# } else if (timerStyle === 'progress-ring') { #>
                            <div class="progress-ring-container">
                                <svg class="progress-ring" width="120" height="120">
                                    <circle class="progress-ring-bg" cx="60" cy="60" r="54"></circle>
                                    <circle class="progress-ring-fill" cx="60" cy="60" r="54" data-unit="seconds"></circle>
                                </svg>
                                <div class="progress-ring-content">
                                    <div class="countdown-number" data-unit="seconds">30</div>
                                </div>
                            </div>
                            <div class="countdown-label">{{{ settings.label_seconds }}}</div>
                        <# } else { #>
                            <# if (labelsPosition === 'top') { #>
                                <div class="countdown-label">{{{ settings.label_seconds }}}</div>
                            <# } #>
                            <div class="countdown-number" data-unit="seconds">30</div>
                            <# if (labelsPosition === 'bottom') { #>
                                <div class="countdown-label">{{{ settings.label_seconds }}}</div>
                            <# } #>
                        <# } #>
                    </div>
                <# } #>

            </div>

            <# if (settings.expiry_action === 'message') { #>
                <div class="countdown-expired-message countdown-hidden">
                    {{{ settings.expiry_message }}}
                </div>
            <# } #>
        </div>
        
        <?php
    }
}
