/**
 * Disable Right Click Widget JavaScript
 * Provides comprehensive content protection functionality
 */

class DisableRightClickWidget {
    constructor(element) {
        this.element = element;
        this.settings = JSON.parse(element.dataset.settings || '{}');
        this.isDevToolsOpen = false;
        this.devToolsCheckInterval = null;
        
        this.init();
    }

    init() {
        // Apply protection based on settings
        if (this.settings.disableRightClick) {
            this.disableRightClick();
        }
        
        if (this.settings.disableTextSelection) {
            this.disableTextSelection();
        }
        
        if (this.settings.disableDragDrop) {
            this.disableDragDrop();
        }
        
        if (this.settings.disableKeyboardShortcuts) {
            this.disableKeyboardShortcuts();
        }
        
        if (this.settings.disableDeveloperTools) {
            this.disableDeveloperTools();
        }
        
        if (this.settings.blurOnDevtools) {
            this.startDevToolsDetection();
        }
        
        // Add protection styles
        this.addProtectionStyles();
    }

    getTargetElements() {
        if (this.settings.protectionScope === 'specific_elements' && this.settings.targetSelectors) {
            const selectors = this.settings.targetSelectors.split(',').map(s => s.trim()).join(', ');
            return document.querySelectorAll(selectors);
        }
        return [document.body];
    }

    disableRightClick() {
        const targets = this.getTargetElements();
        
        targets.forEach(target => {
            target.addEventListener('contextmenu', (e) => {
                e.preventDefault();
                this.showWarning();
                return false;
            });
        });
    }

    disableTextSelection() {
        const targets = this.getTargetElements();
        
        targets.forEach(target => {
            target.style.userSelect = 'none';
            target.style.webkitUserSelect = 'none';
            target.style.mozUserSelect = 'none';
            target.style.msUserSelect = 'none';
            
            // Prevent text selection events
            target.addEventListener('selectstart', (e) => {
                e.preventDefault();
                return false;
            });
            
            target.addEventListener('dragstart', (e) => {
                e.preventDefault();
                return false;
            });
        });
    }

    disableDragDrop() {
        const images = document.querySelectorAll('img');
        
        images.forEach(img => {
            img.addEventListener('dragstart', (e) => {
                e.preventDefault();
                this.showWarning();
                return false;
            });
            
            img.addEventListener('drop', (e) => {
                e.preventDefault();
                return false;
            });
            
            // Disable image saving on mobile
            img.addEventListener('touchstart', (e) => {
                if (e.touches.length > 1) {
                    e.preventDefault();
                }
            });
        });
    }

    disableKeyboardShortcuts() {
        document.addEventListener('keydown', (e) => {
            // Disable common shortcuts
            const blockedKeys = [
                { ctrl: true, key: 'a' }, // Select All
                { ctrl: true, key: 'c' }, // Copy
                { ctrl: true, key: 'v' }, // Paste
                { ctrl: true, key: 'x' }, // Cut
                { ctrl: true, key: 's' }, // Save
                { ctrl: true, key: 'p' }, // Print
                { ctrl: true, key: 'u' }, // View Source
                { key: 'F12' },           // Developer Tools
                { ctrl: true, shift: true, key: 'I' }, // Developer Tools
                { ctrl: true, shift: true, key: 'J' }, // Console
                { ctrl: true, shift: true, key: 'C' }, // Inspector
                { key: 'F5' },            // Refresh (optional)
                { ctrl: true, key: 'r' }, // Refresh (optional)
            ];

            const currentKey = {
                ctrl: e.ctrlKey || e.metaKey,
                shift: e.shiftKey,
                alt: e.altKey,
                key: e.key
            };

            const isBlocked = blockedKeys.some(blocked => {
                return (!blocked.ctrl || currentKey.ctrl) &&
                       (!blocked.shift || currentKey.shift) &&
                       (!blocked.alt || currentKey.alt) &&
                       blocked.key.toLowerCase() === currentKey.key.toLowerCase();
            });

            if (isBlocked) {
                e.preventDefault();
                e.stopPropagation();
                this.showWarning();
                return false;
            }
        });
    }

    disableDeveloperTools() {
        // Detect F12 and other dev tools shortcuts
        document.addEventListener('keydown', (e) => {
            if (e.key === 'F12' || 
                (e.ctrlKey && e.shiftKey && (e.key === 'I' || e.key === 'J' || e.key === 'C')) ||
                (e.ctrlKey && e.key === 'U')) {
                e.preventDefault();
                this.showWarning('Developer tools are disabled!');
                return false;
            }
        });

        // Disable right-click inspect element
        document.addEventListener('contextmenu', (e) => {
            e.preventDefault();
            return false;
        });
    }

    startDevToolsDetection() {
        // Basic dev tools detection (not foolproof but adds a layer)
        this.devToolsCheckInterval = setInterval(() => {
            const widthThreshold = window.outerWidth - window.innerWidth > 160;
            const heightThreshold = window.outerHeight - window.innerHeight > 160;
            
            if (widthThreshold || heightThreshold) {
                if (!this.isDevToolsOpen) {
                    this.isDevToolsOpen = true;
                    this.blurContent();
                }
            } else {
                if (this.isDevToolsOpen) {
                    this.isDevToolsOpen = false;
                    this.unblurContent();
                }
            }
        }, 1000);
    }

    blurContent() {
        document.body.style.filter = 'blur(5px)';
        document.body.style.pointerEvents = 'none';
        
        // Show warning overlay
        const overlay = document.createElement('div');
        overlay.id = 'devtools-warning';
        overlay.innerHTML = `
            <div style="position: fixed; top: 50%; left: 50%; transform: translate(-50%, -50%); 
                        background: #fff; padding: 30px; border-radius: 10px; box-shadow: 0 10px 30px rgba(0,0,0,0.3);
                        text-align: center; z-index: 999999; font-family: Arial, sans-serif;">
                <h3 style="color: #e74c3c; margin: 0 0 15px 0;">⚠️ Developer Tools Detected</h3>
                <p style="margin: 0; color: #555;">Please close developer tools to continue viewing this content.</p>
            </div>
        `;
        overlay.style.cssText = `
            position: fixed; top: 0; left: 0; width: 100%; height: 100%; 
            background: rgba(0,0,0,0.8); z-index: 999998;
        `;
        document.body.appendChild(overlay);
    }

    unblurContent() {
        document.body.style.filter = '';
        document.body.style.pointerEvents = '';
        
        const overlay = document.getElementById('devtools-warning');
        if (overlay) {
            overlay.remove();
        }
    }

    showWarning(customMessage = null) {
        if (!this.settings.showWarningMessage) return;
        
        const message = customMessage || this.settings.warningMessage || 'Content is protected!';
        
        // Create toast notification instead of alert
        const toast = document.createElement('div');
        toast.innerHTML = `🛡️ ${message}`;
        toast.style.cssText = `
            position: fixed; top: 20px; right: 20px; 
            background: #e74c3c; color: white; padding: 15px 20px; 
            border-radius: 5px; z-index: 999999; font-size: 14px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.3);
            animation: slideIn 0.3s ease-out;
        `;
        
        // Add animation keyframes
        if (!document.getElementById('protection-toast-styles')) {
            const style = document.createElement('style');
            style.id = 'protection-toast-styles';
            style.textContent = `
                @keyframes slideIn {
                    from { transform: translateX(100%); opacity: 0; }
                    to { transform: translateX(0); opacity: 1; }
                }
            `;
            document.head.appendChild(style);
        }
        
        document.body.appendChild(toast);
        
        // Remove toast after 3 seconds
        setTimeout(() => {
            if (toast.parentNode) {
                toast.remove();
            }
        }, 3000);
    }

    addProtectionStyles() {
        // Add CSS to prevent common bypass methods
        const style = document.createElement('style');
        style.textContent = `
            .disable-right-click-protection {
                -webkit-touch-callout: none;
                -webkit-user-select: none;
                -khtml-user-select: none;
                -moz-user-select: none;
                -ms-user-select: none;
                user-select: none;
                -webkit-tap-highlight-color: transparent;
            }
            
            .protection-badge {
                position: fixed;
                width: 30px;
                height: 30px;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 12px;
                color: white;
                z-index: 999;
                box-shadow: 0 2px 8px rgba(0,0,0,0.3);
                cursor: default;
                user-select: none;
            }
            
            .protection-badge-top-left { top: 20px; left: 20px; }
            .protection-badge-top-right { top: 20px; right: 20px; }
            .protection-badge-bottom-left { bottom: 20px; left: 20px; }
            .protection-badge-bottom-right { bottom: 20px; right: 20px; }
            
            /* Disable text selection highlighting */
            ::selection { background: transparent; }
            ::-moz-selection { background: transparent; }
        `;
        document.head.appendChild(style);
        
        // Apply protection class to target elements
        const targets = this.getTargetElements();
        targets.forEach(target => {
            target.classList.add('disable-right-click-protection');
        });
    }

    // Cleanup method
    destroy() {
        if (this.devToolsCheckInterval) {
            clearInterval(this.devToolsCheckInterval);
        }
        this.unblurContent();
    }
}

// Initialize widgets when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    const widgets = document.querySelectorAll('.disable-right-click-widget');
    
    widgets.forEach(widget => {
        new DisableRightClickWidget(widget);
    });
});

// Re-initialize after Elementor preview updates
if (typeof elementorFrontend !== 'undefined') {
    elementorFrontend.hooks.addAction('frontend/element_ready/disable_right_click.default', function($scope) {
        const widget = $scope.find('.disable-right-click-widget')[0];
        if (widget) {
            new DisableRightClickWidget(widget);
        }
    });
}

// Expose class globally for manual initialization
window.DisableRightClickWidget = DisableRightClickWidget;
