<?php
/**
 * Disable Right Click Widget for Elementor
 * Blocks right-click actions and keyboard shortcuts to protect content
 *
 * @package RelayoutDesigns
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Make sure Elementor is active
if (!class_exists('\Elementor\Widget_Base')) {
    return;
}

class Disable_Right_Click_Widget extends \Elementor\Widget_Base {

    public function get_name() {
        return 'disable_right_click';
    }

    public function get_title() {
        return __('Disable Right Click', 'relayout-design');
    }

    public function get_icon() {
        return 'eicon-lock';
    }

    public function get_categories() {
        return ['relayout-designs'];
    }

    public function get_script_depends() {
        return ['disable-right-click-widget-script'];
    }

    public function get_style_depends() {
        return ['disable-right-click-widget-style'];
    }

    protected function register_controls() {
        // Protection Settings Section
        $this->start_controls_section(
            'protection_section',
            [
                'label' => __('Protection Settings', 'relayout-design'),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'protection_notice',
            [
                'type' => \Elementor\Controls_Manager::RAW_HTML,
                'raw' => '<div style="background: #e3f2fd; padding: 15px; border-radius: 5px; border-left: 4px solid #2196f3;">
                    <h4 style="margin: 0 0 10px 0; color: #1976d2;">🛡️ Content Protection Active</h4>
                    <p style="margin: 0; color: #555;">This widget will protect your entire page content from right-click, copy, and save actions. The protection is active on the frontend only.</p>
                </div>',
            ]
        );

        $this->add_control(
            'disable_right_click',
            [
                'label' => __('Disable Right Click', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
                'description' => __('Prevents context menu from appearing on right-click', 'relayout-design'),
            ]
        );

        $this->add_control(
            'disable_text_selection',
            [
                'label' => __('Disable Text Selection', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
                'description' => __('Prevents users from selecting text on the page', 'relayout-design'),
            ]
        );

        $this->add_control(
            'disable_drag_drop',
            [
                'label' => __('Disable Image Drag & Drop', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
                'description' => __('Prevents dragging images to desktop or other applications', 'relayout-design'),
            ]
        );

        $this->add_control(
            'disable_keyboard_shortcuts',
            [
                'label' => __('Disable Keyboard Shortcuts', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
                'description' => __('Blocks Ctrl+A, Ctrl+C, Ctrl+V, Ctrl+S, F12, etc.', 'relayout-design'),
            ]
        );

        $this->add_control(
            'disable_developer_tools',
            [
                'label' => __('Disable Developer Tools', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
                'description' => __('Attempts to block F12, Ctrl+Shift+I, and other dev tools shortcuts', 'relayout-design'),
            ]
        );

        $this->add_control(
            'show_warning_message',
            [
                'label' => __('Show Warning Message', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
                'description' => __('Display alert when protection is triggered', 'relayout-design'),
            ]
        );

        $this->add_control(
            'warning_message',
            [
                'label' => __('Warning Message', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Content is protected! Right-click is disabled.', 'relayout-design'),
                'condition' => [
                    'show_warning_message' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        // Advanced Settings Section
        $this->start_controls_section(
            'advanced_section',
            [
                'label' => __('Advanced Settings', 'relayout-design'),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'protection_scope',
            [
                'label' => __('Protection Scope', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'entire_page',
                'options' => [
                    'entire_page' => __('Entire Page', 'relayout-design'),
                    'specific_elements' => __('Specific Elements Only', 'relayout-design'),
                ],
                'description' => __('Choose whether to protect the entire page or only specific elements', 'relayout-design'),
            ]
        );

        $this->add_control(
            'target_selectors',
            [
                'label' => __('Target CSS Selectors', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => '.protected-content, .elementor-widget-image',
                'placeholder' => '.my-images, #content-area, .gallery',
                'description' => __('CSS selectors for elements to protect (comma-separated)', 'relayout-design'),
                'condition' => [
                    'protection_scope' => 'specific_elements',
                ],
            ]
        );

        $this->add_control(
            'exclude_admin',
            [
                'label' => __('Exclude Administrators', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
                'description' => __('Disable protection for logged-in administrators', 'relayout-design'),
            ]
        );

        $this->add_control(
            'blur_on_devtools',
            [
                'label' => __('Blur Content on DevTools', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => '',
                'description' => __('Blur page content when developer tools are detected', 'relayout-design'),
            ]
        );

        $this->end_controls_section();

        // Style Section
        $this->start_controls_section(
            'style_section',
            [
                'label' => __('Protection Indicator Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'show_protection_badge',
            [
                'label' => __('Show Protection Badge', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => '',
                'description' => __('Display a small protection indicator on the page', 'relayout-design'),
            ]
        );

        $this->add_control(
            'badge_position',
            [
                'label' => __('Badge Position', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'bottom-right',
                'options' => [
                    'top-left' => __('Top Left', 'relayout-design'),
                    'top-right' => __('Top Right', 'relayout-design'),
                    'bottom-left' => __('Bottom Left', 'relayout-design'),
                    'bottom-right' => __('Bottom Right', 'relayout-design'),
                ],
                'condition' => [
                    'show_protection_badge' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'badge_color',
            [
                'label' => __('Badge Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ff5722',
                'condition' => [
                    'show_protection_badge' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        
        // Check if admin should be excluded
        if ($settings['exclude_admin'] === 'yes' && current_user_can('administrator')) {
            echo '<div class="protection-disabled-admin" style="padding: 10px; background: #fff3cd; border: 1px solid #ffeaa7; border-radius: 4px; color: #856404; text-align: center; font-size: 12px;">
                🛡️ Content Protection is disabled for administrators
            </div>';
            return;
        }

        // Prepare settings for JavaScript
        $protection_settings = [
            'disableRightClick' => $settings['disable_right_click'] === 'yes',
            'disableTextSelection' => $settings['disable_text_selection'] === 'yes',
            'disableDragDrop' => $settings['disable_drag_drop'] === 'yes',
            'disableKeyboardShortcuts' => $settings['disable_keyboard_shortcuts'] === 'yes',
            'disableDeveloperTools' => $settings['disable_developer_tools'] === 'yes',
            'showWarningMessage' => $settings['show_warning_message'] === 'yes',
            'warningMessage' => $settings['warning_message'],
            'protectionScope' => $settings['protection_scope'],
            'targetSelectors' => $settings['target_selectors'] ?? '',
            'blurOnDevtools' => $settings['blur_on_devtools'] === 'yes',
        ];
        ?>
        
        <div class="disable-right-click-widget" 
             data-settings="<?php echo esc_attr(json_encode($protection_settings)); ?>">
            
            <?php if ($settings['show_protection_badge'] === 'yes'): ?>
                <div class="protection-badge protection-badge-<?php echo esc_attr($settings['badge_position']); ?>" 
                     style="background-color: <?php echo esc_attr($settings['badge_color']); ?>">
                    🛡️
                </div>
            <?php endif; ?>
            
            <div class="protection-status" style="display: none;">
                <span class="protection-active">🛡️ Content Protection Active</span>
            </div>
        </div>
        
        <?php
    }
}
