/**
 * FAQ Accordion Widget JavaScript
 * Handles accordion functionality, animations, and expand/collapse all features
 */

class FAQAccordionWidget {
    constructor(element) {
        this.element = element;
        this.container = element.querySelector('.faq-container');
        this.items = element.querySelectorAll('.faq-item');
        this.expandCollapseBtn = element.querySelector('.faq-expand-collapse');
        this.faqData = element.querySelector('.faq-data');
        
        this.accordionType = element.dataset.accordionType || 'single';
        this.animationSpeed = parseInt(element.dataset.animationSpeed) || 300;
        
        this.expandText = '';
        this.collapseText = '';
        this.iconType = '';
        this.customOpenIcon = '';
        this.customCloseIcon = '';
        
        this.init();
    }
    
    init() {
        if (!this.container || this.items.length === 0) return;
        
        this.parseData();
        this.bindEvents();
        this.initializeItems();
    }
    
    parseData() {
        if (!this.faqData) return;
        
        const expandTextSpan = this.faqData.querySelector('[data-expand-text]');
        const collapseTextSpan = this.faqData.querySelector('[data-collapse-text]');
        const iconTypeSpan = this.faqData.querySelector('[data-icon-type]');
        const customOpenIconSpan = this.faqData.querySelector('[data-custom-open-icon]');
        const customCloseIconSpan = this.faqData.querySelector('[data-custom-close-icon]');
        
        this.expandText = expandTextSpan ? expandTextSpan.dataset.expandText : 'Expand All';
        this.collapseText = collapseTextSpan ? collapseTextSpan.dataset.collapseText : 'Collapse All';
        this.iconType = iconTypeSpan ? iconTypeSpan.dataset.iconType : 'plus_minus';
        this.customOpenIcon = customOpenIconSpan ? customOpenIconSpan.dataset.customOpenIcon : '';
        this.customCloseIcon = customCloseIconSpan ? customCloseIconSpan.dataset.customCloseIcon : '';
    }
    
    bindEvents() {
        // Bind header click events
        this.items.forEach((item, index) => {
            const header = item.querySelector('.faq-item-header');
            if (header) {
                header.addEventListener('click', (e) => {
                    e.preventDefault();
                    this.toggleItem(index);
                });
                
                // Keyboard support
                header.addEventListener('keydown', (e) => {
                    if (e.key === 'Enter' || e.key === ' ') {
                        e.preventDefault();
                        this.toggleItem(index);
                    }
                });
            }
        });
        
        // Bind expand/collapse all button
        if (this.expandCollapseBtn) {
            this.expandCollapseBtn.addEventListener('click', (e) => {
                e.preventDefault();
                this.toggleAll();
            });
        }
    }
    
    initializeItems() {
        this.items.forEach((item, index) => {
            const content = item.querySelector('.faq-content');
            const header = item.querySelector('.faq-item-header');
            
            if (!content || !header) return;
            
            const isActive = item.classList.contains('active');
            
            // Set initial state
            if (isActive) {
                content.style.display = 'block';
                content.style.maxHeight = content.scrollHeight + 'px';
                header.setAttribute('aria-expanded', 'true');
            } else {
                content.style.display = 'none';
                content.style.maxHeight = '0px';
                header.setAttribute('aria-expanded', 'false');
            }
            
            // Set initial icon
            this.updateIcon(item, isActive);
        });
        
        this.updateExpandCollapseButton();
    }
    
    toggleItem(index) {
        const item = this.items[index];
        if (!item) return;
        
        const isCurrentlyActive = item.classList.contains('active');
        
        // If single accordion type, close all other items first
        if (this.accordionType === 'single' && !isCurrentlyActive) {
            this.items.forEach((otherItem, otherIndex) => {
                if (otherIndex !== index && otherItem.classList.contains('active')) {
                    this.closeItem(otherIndex);
                }
            });
        }
        
        // Toggle current item
        if (isCurrentlyActive) {
            this.closeItem(index);
        } else {
            this.openItem(index);
        }
        
        this.updateExpandCollapseButton();
        
        // Dispatch custom event
        this.element.dispatchEvent(new CustomEvent('faqToggled', {
            detail: {
                index,
                isOpen: !isCurrentlyActive,
                accordionType: this.accordionType
            }
        }));
    }
    
    openItem(index) {
        const item = this.items[index];
        if (!item) return;
        
        const content = item.querySelector('.faq-content');
        const header = item.querySelector('.faq-item-header');
        
        if (!content || !header) return;
        
        item.classList.add('active');
        header.setAttribute('aria-expanded', 'true');
        
        // Show content and animate
        content.style.display = 'block';
        const targetHeight = content.scrollHeight;
        
        // Start animation
        content.style.maxHeight = '0px';
        content.style.overflow = 'hidden';
        
        requestAnimationFrame(() => {
            content.style.transition = `max-height ${this.animationSpeed}ms ease-out`;
            content.style.maxHeight = targetHeight + 'px';
            
            setTimeout(() => {
                content.style.maxHeight = 'none';
                content.style.overflow = 'visible';
                content.style.transition = '';
            }, this.animationSpeed);
        });
        
        this.updateIcon(item, true);
    }
    
    closeItem(index) {
        const item = this.items[index];
        if (!item) return;
        
        const content = item.querySelector('.faq-content');
        const header = item.querySelector('.faq-item-header');
        
        if (!content || !header) return;
        
        item.classList.remove('active');
        header.setAttribute('aria-expanded', 'false');
        
        // Animate close
        const currentHeight = content.scrollHeight;
        content.style.maxHeight = currentHeight + 'px';
        content.style.overflow = 'hidden';
        
        requestAnimationFrame(() => {
            content.style.transition = `max-height ${this.animationSpeed}ms ease-out`;
            content.style.maxHeight = '0px';
            
            setTimeout(() => {
                content.style.display = 'none';
                content.style.transition = '';
                content.style.overflow = 'visible';
            }, this.animationSpeed);
        });
        
        this.updateIcon(item, false);
    }
    
    updateIcon(item, isOpen) {
        const icon = item.querySelector('.faq-icon i');
        if (!icon) return;
        
        let iconClass = '';
        
        switch (this.iconType) {
            case 'plus_minus':
                iconClass = isOpen ? 'fas fa-minus' : 'fas fa-plus';
                break;
            case 'chevron':
                iconClass = isOpen ? 'fas fa-chevron-up' : 'fas fa-chevron-down';
                break;
            case 'arrow':
                iconClass = isOpen ? 'fas fa-arrow-up' : 'fas fa-arrow-down';
                break;
            case 'custom':
                if (isOpen && this.customCloseIcon) {
                    iconClass = this.customCloseIcon;
                } else if (!isOpen && this.customOpenIcon) {
                    iconClass = this.customOpenIcon;
                } else {
                    iconClass = 'fas fa-plus';
                }
                break;
        }
        
        if (iconClass) {
            icon.className = iconClass;
        }
    }
    
    toggleAll() {
        if (!this.expandCollapseBtn) return;
        
        const action = this.expandCollapseBtn.dataset.action;
        
        if (action === 'expand') {
            this.expandAll();
        } else {
            this.collapseAll();
        }
        
        this.updateExpandCollapseButton();
    }
    
    expandAll() {
        this.items.forEach((item, index) => {
            if (!item.classList.contains('active')) {
                setTimeout(() => {
                    this.openItem(index);
                }, index * 100); // Stagger animation
            }
        });
        
        // Dispatch custom event
        this.element.dispatchEvent(new CustomEvent('faqExpandedAll', {
            detail: { totalItems: this.items.length }
        }));
    }
    
    collapseAll() {
        this.items.forEach((item, index) => {
            if (item.classList.contains('active')) {
                setTimeout(() => {
                    this.closeItem(index);
                }, index * 50); // Stagger animation
            }
        });
        
        // Dispatch custom event
        this.element.dispatchEvent(new CustomEvent('faqCollapsedAll', {
            detail: { totalItems: this.items.length }
        }));
    }
    
    updateExpandCollapseButton() {
        if (!this.expandCollapseBtn) return;
        
        const activeItems = this.element.querySelectorAll('.faq-item.active');
        const allItemsOpen = activeItems.length === this.items.length;
        
        if (allItemsOpen) {
            this.expandCollapseBtn.textContent = this.collapseText;
            this.expandCollapseBtn.dataset.action = 'collapse';
        } else {
            this.expandCollapseBtn.textContent = this.expandText;
            this.expandCollapseBtn.dataset.action = 'expand';
        }
    }
    
    // Public methods
    openItemByIndex(index) {
        if (index >= 0 && index < this.items.length) {
            this.openItem(index);
        }
    }
    
    closeItemByIndex(index) {
        if (index >= 0 && index < this.items.length) {
            this.closeItem(index);
        }
    }
    
    getActiveItems() {
        return Array.from(this.items).filter(item => item.classList.contains('active'));
    }
    
    destroy() {
        // Remove event listeners and clean up
        this.items.forEach(item => {
            const header = item.querySelector('.faq-item-header');
            if (header) {
                header.replaceWith(header.cloneNode(true));
            }
        });
        
        if (this.expandCollapseBtn) {
            this.expandCollapseBtn.replaceWith(this.expandCollapseBtn.cloneNode(true));
        }
    }
}

// Initialize widgets when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    initializeFAQAccordionWidgets();
});

// Initialize for Elementor editor
jQuery(window).on('elementor/frontend/init', function() {
    elementorFrontend.hooks.addAction('frontend/element_ready/faq_accordion_widget.default', function($scope) {
        initializeFAQAccordionWidgets($scope[0]);
    });
});

function initializeFAQAccordionWidgets(container = document) {
    const widgets = container.querySelectorAll('.faq-accordion-widget');
    
    widgets.forEach(widget => {
        // Avoid double initialization
        if (widget.faqAccordionInstance) {
            widget.faqAccordionInstance.destroy();
        }
        
        widget.faqAccordionInstance = new FAQAccordionWidget(widget);
    });
}

// Handle dynamic content (AJAX, SPA)
if (window.MutationObserver) {
    const observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                mutation.addedNodes.forEach(function(node) {
                    if (node.nodeType === 1) { // Element node
                        const widgets = node.querySelectorAll ? node.querySelectorAll('.faq-accordion-widget') : [];
                        if (widgets.length > 0) {
                            initializeFAQAccordionWidgets(node);
                        }
                        
                        if (node.classList && node.classList.contains('faq-accordion-widget')) {
                            initializeFAQAccordionWidgets(node.parentElement);
                        }
                    }
                });
            }
        });
    });
    
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
}

// Global API
window.FAQAccordionWidget = FAQAccordionWidget;
window.initializeFAQAccordionWidgets = initializeFAQAccordionWidgets;

// Utility functions for developers
window.faqAccordion = {
    expandAll: function(widgetElement) {
        if (widgetElement && widgetElement.faqAccordionInstance) {
            widgetElement.faqAccordionInstance.expandAll();
        }
    },
    
    collapseAll: function(widgetElement) {
        if (widgetElement && widgetElement.faqAccordionInstance) {
            widgetElement.faqAccordionInstance.collapseAll();
        }
    },
    
    openItem: function(widgetElement, index) {
        if (widgetElement && widgetElement.faqAccordionInstance) {
            widgetElement.faqAccordionInstance.openItemByIndex(index);
        }
    },
    
    closeItem: function(widgetElement, index) {
        if (widgetElement && widgetElement.faqAccordionInstance) {
            widgetElement.faqAccordionInstance.closeItemByIndex(index);
        }
    },
    
    getActiveItems: function(widgetElement) {
        if (widgetElement && widgetElement.faqAccordionInstance) {
            return widgetElement.faqAccordionInstance.getActiveItems();
        }
        return [];
    }
};
