<?php
/**
 * FAQ Accordion Widget for Elementor
 * 
 * Creates an accordion-style FAQ section with schema.org structured data
 * Includes expand/collapse functionality, custom icons, and styling options
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Make sure Elementor is active
if (!class_exists('\Elementor\Widget_Base')) {
    return;
}

class FAQ_Accordion_Widget extends \Elementor\Widget_Base {

    public function get_name() {
        return 'faq_accordion_widget';
    }

    public function get_title() {
        return __('FAQ Accordion', 'relayout-design');
    }

    public function get_icon() {
        return 'eicon-accordion';
    }

    public function get_categories() {
        return ['relayout-designs'];
    }

    public function get_keywords() {
        return ['faq', 'accordion', 'questions', 'answers', 'schema', 'seo'];
    }

    protected function register_controls() {
        // Content Section
        $this->start_controls_section(
            'content_section',
            [
                'label' => __('FAQ Items', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'question',
            [
                'label' => __('Question', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('What is your question?', 'relayout-design'),
                'placeholder' => __('Enter your question here', 'relayout-design'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'answer',
            [
                'label' => __('Answer', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::WYSIWYG,
                'default' => __('This is the answer to your question. You can add rich text, links, and formatting here.', 'relayout-design'),
                'placeholder' => __('Enter your answer here', 'relayout-design'),
                'show_label' => false,
            ]
        );

        $repeater->add_control(
            'is_active',
            [
                'label' => __('Open by Default', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'relayout-design'),
                'label_off' => __('No', 'relayout-design'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'faq_items',
            [
                'label' => __('FAQ Items', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'question' => __('What is an FAQ section?', 'relayout-design'),
                        'answer' => __('An FAQ section contains frequently asked questions and their answers to help users find information quickly.', 'relayout-design'),
                        'is_active' => 'yes',
                    ],
                    [
                        'question' => __('How do I add more questions?', 'relayout-design'),
                        'answer' => __('You can add more questions by clicking the "Add Item" button and filling in the question and answer fields.', 'relayout-design'),
                        'is_active' => 'no',
                    ],
                    [
                        'question' => __('Does this support SEO?', 'relayout-design'),
                        'answer' => __('Yes! This widget automatically generates schema.org FAQ structured data for better SEO performance.', 'relayout-design'),
                        'is_active' => 'no',
                    ],
                ],
                'title_field' => '{{{ question }}}',
            ]
        );

        $this->end_controls_section();

        // Settings Section
        $this->start_controls_section(
            'settings_section',
            [
                'label' => __('Accordion Settings', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'accordion_type',
            [
                'label' => __('Accordion Type', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'single',
                'options' => [
                    'single' => __('Single (One at a time)', 'relayout-design'),
                    'multiple' => __('Multiple (Multiple open)', 'relayout-design'),
                ],
            ]
        );

        $this->add_control(
            'icon_type',
            [
                'label' => __('Icon Type', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'plus_minus',
                'options' => [
                    'plus_minus' => __('Plus/Minus', 'relayout-design'),
                    'chevron' => __('Chevron', 'relayout-design'),
                    'arrow' => __('Arrow', 'relayout-design'),
                    'custom' => __('Custom Icons', 'relayout-design'),
                    'none' => __('No Icon', 'relayout-design'),
                ],
            ]
        );

        $this->add_control(
            'custom_open_icon',
            [
                'label' => __('Custom Open Icon', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-plus',
                    'library' => 'fa-solid',
                ],
                'condition' => [
                    'icon_type' => 'custom',
                ],
            ]
        );

        $this->add_control(
            'custom_close_icon',
            [
                'label' => __('Custom Close Icon', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-minus',
                    'library' => 'fa-solid',
                ],
                'condition' => [
                    'icon_type' => 'custom',
                ],
            ]
        );

        $this->add_control(
            'icon_position',
            [
                'label' => __('Icon Position', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'right',
                'options' => [
                    'left' => __('Left', 'relayout-design'),
                    'right' => __('Right', 'relayout-design'),
                ],
                'condition' => [
                    'icon_type!' => 'none',
                ],
            ]
        );

        $this->add_control(
            'animation_speed',
            [
                'label' => __('Animation Speed (ms)', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 100,
                        'max' => 1000,
                        'step' => 50,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 300,
                ],
            ]
        );

        $this->add_control(
            'show_expand_collapse',
            [
                'label' => __('Show Expand/Collapse All', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'relayout-design'),
                'label_off' => __('Hide', 'relayout-design'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'expand_text',
            [
                'label' => __('Expand All Text', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Expand All', 'relayout-design'),
                'condition' => [
                    'show_expand_collapse' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'collapse_text',
            [
                'label' => __('Collapse All Text', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Collapse All', 'relayout-design'),
                'condition' => [
                    'show_expand_collapse' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'enable_schema',
            [
                'label' => __('Enable Schema.org FAQ', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'relayout-design'),
                'label_off' => __('No', 'relayout-design'),
                'return_value' => 'yes',
                'default' => 'yes',
                'description' => __('Adds structured data for better SEO', 'relayout-design'),
            ]
        );

        $this->end_controls_section();

        // Add style sections here (truncated for brevity)
        $this->add_style_sections();
    }

    private function add_style_sections() {
        // Question Style Section
        $this->start_controls_section(
            'question_style_section',
            [
                'label' => __('Question Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'question_typography',
                'selector' => '{{WRAPPER}} .faq-question',
            ]
        );

        $this->add_control(
            'question_text_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}} .faq-question' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'question_background',
                'selector' => '{{WRAPPER}} .faq-item-header',
            ]
        );

        $this->add_responsive_control(
            'question_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .faq-item-header' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Answer Style Section
        $this->start_controls_section(
            'answer_style_section',
            [
                'label' => __('Answer Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'answer_typography',
                'selector' => '{{WRAPPER}} .faq-answer',
            ]
        );

        $this->add_control(
            'answer_text_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#666666',
                'selectors' => [
                    '{{WRAPPER}} .faq-answer' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'answer_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .faq-answer' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Icon Style Section
        $this->start_controls_section(
            'icon_style_section',
            [
                'label' => __('Icon Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'icon_type!' => 'none',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label' => __('Icon Size', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 50,
                    ],
                ],
                'default' => [
                    'size' => 16,
                ],
                'selectors' => [
                    '{{WRAPPER}} .faq-icon' => 'font-size: {{SIZE}}px;',
                ],
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label' => __('Icon Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#666666',
                'selectors' => [
                    '{{WRAPPER}} .faq-icon' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function get_icon_html($icon_type, $is_open = false, $custom_open_icon = null, $custom_close_icon = null) {
        $icon_class = '';

        switch ($icon_type) {
            case 'plus_minus':
                $icon_class = $is_open ? 'fas fa-minus' : 'fas fa-plus';
                break;
            case 'chevron':
                $icon_class = $is_open ? 'fas fa-chevron-up' : 'fas fa-chevron-down';
                break;
            case 'arrow':
                $icon_class = $is_open ? 'fas fa-arrow-up' : 'fas fa-arrow-down';
                break;
            case 'custom':
                if ($is_open && $custom_close_icon && !empty($custom_close_icon['value'])) {
                    $icon_class = $custom_close_icon['value'];
                } elseif (!$is_open && $custom_open_icon && !empty($custom_open_icon['value'])) {
                    $icon_class = $custom_open_icon['value'];
                } else {
                    $icon_class = 'fas fa-plus';
                }
                break;
            default:
                return '';
        }

        return $icon_class ? '<span class="faq-icon"><i class="' . esc_attr($icon_class) . '"></i></span>' : '';
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        
        if (empty($settings['faq_items'])) {
            return;
        }

        $this->add_render_attribute('wrapper', 'class', 'faq-accordion-widget');
        $this->add_render_attribute('wrapper', 'data-accordion-type', $settings['accordion_type']);
        $this->add_render_attribute('wrapper', 'data-animation-speed', $settings['animation_speed']['size']);

        // Generate Schema.org FAQ structured data
        $schema_data = [];
        if ($settings['enable_schema'] === 'yes') {
            $schema_data = [
                '@context' => 'https://schema.org',
                '@type' => 'FAQPage',
                'mainEntity' => []
            ];

            foreach ($settings['faq_items'] as $item) {
                $schema_data['mainEntity'][] = [
                    '@type' => 'Question',
                    'name' => wp_strip_all_tags($item['question']),
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => wp_strip_all_tags($item['answer'])
                    ]
                ];
            }
        }

        ?>
        <div <?php echo $this->get_render_attribute_string('wrapper'); ?>>
            <?php if ($settings['show_expand_collapse'] === 'yes') : ?>
                <div class="faq-controls">
                    <button class="faq-expand-collapse" data-action="expand">
                        <?php echo esc_html($settings['expand_text']); ?>
                    </button>
                </div>
            <?php endif; ?>

            <div class="faq-container">
                <?php foreach ($settings['faq_items'] as $index => $item) : ?>
                    <?php
                    $is_active = $item['is_active'] === 'yes';
                    $item_classes = 'faq-item';
                    if ($is_active) {
                        $item_classes .= ' active';
                    }
                    
                    $header_classes = 'faq-item-header icon-' . esc_attr($settings['icon_position']);
                    ?>
                    <div class="<?php echo esc_attr($item_classes); ?>" data-index="<?php echo esc_attr($index); ?>">
                        <div class="<?php echo esc_attr($header_classes); ?>" role="button" tabindex="0" aria-expanded="<?php echo $is_active ? 'true' : 'false'; ?>" data-icon-position="<?php echo esc_attr($settings['icon_position']); ?>">
                            <?php if ($settings['icon_type'] !== 'none') : ?>
                                <?php echo $this->get_icon_html($settings['icon_type'], $is_active, $settings['custom_open_icon'], $settings['custom_close_icon']); ?>
                            <?php endif; ?>
                            
                            <h3 class="faq-question"><?php echo esc_html($item['question']); ?></h3>
                        </div>
                        
                        <div class="faq-content" style="<?php echo $is_active ? '' : 'display: none;'; ?>">
                            <div class="faq-answer"><?php echo wp_kses_post($item['answer']); ?></div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>

            <?php if (!empty($schema_data)) : ?>
                <script type="application/ld+json">
                    <?php echo wp_json_encode($schema_data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE); ?>
                </script>
            <?php endif; ?>

            <!-- Hidden data for JavaScript -->
            <div class="faq-data" style="display: none;">
                <span data-expand-text="<?php echo esc_attr($settings['expand_text']); ?>"></span>
                <span data-collapse-text="<?php echo esc_attr($settings['collapse_text']); ?>"></span>
                <span data-icon-type="<?php echo esc_attr($settings['icon_type']); ?>"></span>
                <span data-custom-open-icon="<?php echo esc_attr($settings['custom_open_icon']['value'] ?? ''); ?>"></span>
                <span data-custom-close-icon="<?php echo esc_attr($settings['custom_close_icon']['value'] ?? ''); ?>"></span>
            </div>
        </div>
        <?php
    }

    protected function content_template() {
        ?>
        <#
        if (!settings.faq_items || settings.faq_items.length === 0) {
            return;
        }

        var accordionType = settings.accordion_type || 'single';
        var animationSpeed = settings.animation_speed ? settings.animation_speed.size : 300;
        var iconType = settings.icon_type || 'plus_minus';
        var iconPosition = settings.icon_position || 'right';
        
        function getIconClass(iconType, isOpen, customOpenIcon, customCloseIcon) {
            var iconClass = '';
            switch (iconType) {
                case 'plus_minus':
                    iconClass = isOpen ? 'fas fa-minus' : 'fas fa-plus';
                    break;
                case 'chevron':
                    iconClass = isOpen ? 'fas fa-chevron-up' : 'fas fa-chevron-down';
                    break;
                case 'arrow':
                    iconClass = isOpen ? 'fas fa-arrow-up' : 'fas fa-arrow-down';
                    break;
                case 'custom':
                    if (isOpen && customCloseIcon && customCloseIcon.value) {
                        iconClass = customCloseIcon.value;
                    } else if (!isOpen && customOpenIcon && customOpenIcon.value) {
                        iconClass = customOpenIcon.value;
                    } else {
                        iconClass = 'fas fa-plus';
                    }
                    break;
                default:
                    return '';
            }
            return iconClass ? '<span class="faq-icon"><i class="' + iconClass + '"></i></span>' : '';
        }
        #>
        
        <div class="faq-accordion-widget" data-accordion-type="{{ accordionType }}" data-animation-speed="{{ animationSpeed }}">
            <# if (settings.show_expand_collapse === 'yes') { #>
                <div class="faq-controls">
                    <button class="faq-expand-collapse" data-action="expand">
                        {{{ settings.expand_text || 'Expand All' }}}
                    </button>
                </div>
            <# } #>

            <div class="faq-container">
                <# _.each(settings.faq_items, function(item, index) {
                    var isActive = item.is_active === 'yes';
                    var itemClasses = 'faq-item' + (isActive ? ' active' : '');
                    var headerClasses = 'faq-item-header icon-' + iconPosition;
                    var iconHtml = getIconClass(iconType, isActive, settings.custom_open_icon, settings.custom_close_icon);
                #>
                    <div class="{{ itemClasses }}" data-index="{{ index }}">
                        <div class="{{ headerClasses }}" role="button" tabindex="0" aria-expanded="{{ isActive ? 'true' : 'false' }}" data-icon-position="{{ iconPosition }}">
                            <# if (iconType !== 'none') { #>
                                {{{ iconHtml }}}
                            <# } #>
                            
                            <h3 class="faq-question">{{{ item.question }}}</h3>
                        </div>
                        
                        <div class="faq-content" style="{{ isActive ? '' : 'display: none;' }}">
                            <div class="faq-answer">{{{ item.answer }}}</div>
                        </div>
                    </div>
                <# }); #>
            </div>

            <!-- Hidden data for JavaScript -->
            <div class="faq-data" style="display: none;">
                <span data-expand-text="{{ settings.expand_text || 'Expand All' }}"></span>
                <span data-collapse-text="{{ settings.collapse_text || 'Collapse All' }}"></span>
                <span data-icon-type="{{ iconType }}"></span>
                <span data-custom-open-icon="{{ settings.custom_open_icon ? settings.custom_open_icon.value : '' }}"></span>
                <span data-custom-close-icon="{{ settings.custom_close_icon ? settings.custom_close_icon.value : '' }}"></span>
            </div>
        </div>
        <?php
    }
}
