/**
 * Instagram Embed Carousel Widget JavaScript
 * Handles carousel functionality for Instagram embeds
 */

class InstagramFeedWidget {
    constructor(element) {
        this.element = element;
        this.settings = JSON.parse(element.dataset.settings || '{}');
        this.carousel = element.querySelector('.instagram-carousel');
        this.currentSlide = 0;
        this.slides = [];
        this.autoplayInterval = null;
        
        this.init();
    }

    init() {
        if (this.settings.layout === 'carousel') {
            this.initCarousel();
        } else if (this.settings.layout === 'multi-carousel') {
            this.initMultiCarousel();
        }
        
        // Load Instagram embed script
        this.loadInstagramScript();
    }

    initCarousel() {
        if (!this.carousel) return;
        
        this.slides = this.carousel.querySelectorAll('.carousel-slide');
        const prevBtn = this.carousel.querySelector('.carousel-prev');
        const nextBtn = this.carousel.querySelector('.carousel-next');
        const indicators = this.carousel.querySelectorAll('.indicator');
        
        if (this.slides.length <= 1) return;
        
        // Navigation buttons
        const prevBtns = this.carousel.querySelectorAll('.carousel-prev');
        const nextBtns = this.carousel.querySelectorAll('.carousel-next');
        
        prevBtns.forEach(btn => btn.addEventListener('click', () => this.prevSlide()));
        nextBtns.forEach(btn => btn.addEventListener('click', () => this.nextSlide()));
        
        // Indicators
        indicators.forEach((indicator, index) => {
            indicator.addEventListener('click', () => this.goToSlide(index));
        });
        
        // Autoplay
        if (this.settings.autoplay) {
            this.startAutoplay();
            
            // Pause on hover
            this.carousel.addEventListener('mouseenter', () => this.stopAutoplay());
            this.carousel.addEventListener('mouseleave', () => this.startAutoplay());
        }
        
        // Touch/swipe support
        this.initTouchSupport();
        
        // Initialize layout specific features
        this.initLayoutSpecific();
    }
    
    initMultiCarousel() {
        this.multiCarousel = this.element.querySelector('.instagram-multi-carousel');
        if (!this.multiCarousel) return;
        
        this.multiTrack = this.multiCarousel.querySelector('.multi-carousel-track');
        this.multiItems = this.multiCarousel.querySelectorAll('.multi-carousel-item');
        const prevBtn = this.multiCarousel.querySelector('.carousel-prev');
        const nextBtn = this.multiCarousel.querySelector('.carousel-next');
        
        this.itemsToShow = parseInt(this.settings.multiCarouselItems) || 3;
        this.scrollItems = parseInt(this.settings.multiCarouselScroll) || 1;
        this.currentPosition = 0;
        this.maxPosition = Math.max(0, this.multiItems.length - this.itemsToShow);
        
        if (this.multiItems.length <= this.itemsToShow) return;
        
        // Navigation buttons
        prevBtn?.addEventListener('click', () => this.prevMultiSlide());
        nextBtn?.addEventListener('click', () => this.nextMultiSlide());
        
        // Autoplay
        if (this.settings.autoplay) {
            this.startMultiAutoplay();
            
            // Pause on hover
            this.multiCarousel.addEventListener('mouseenter', () => this.stopAutoplay());
            this.multiCarousel.addEventListener('mouseleave', () => this.startMultiAutoplay());
        }
        
        // Touch/swipe support for multi-carousel
        this.initMultiTouchSupport();
        
        // Set initial position
        this.updateMultiCarouselPosition();
    }
    
    initLayoutSpecific() {
        if (this.settings.layout === 'masonry') {
            this.initMasonry();
        } else if (this.settings.layout === 'pinterest') {
            this.initPinterest();
        }
    }
    
    initMasonry() {
        const masonryContainer = this.element.querySelector('.instagram-masonry');
        if (!masonryContainer) return;
        
        // Simple masonry layout using CSS columns
        const items = masonryContainer.querySelectorAll('.instagram-embed-container');
        
        // Wait for images to load before applying masonry
        let loadedCount = 0;
        const totalItems = items.length;
        
        items.forEach(item => {
            const iframe = item.querySelector('iframe');
            if (iframe) {
                iframe.addEventListener('load', () => {
                    loadedCount++;
                    if (loadedCount === totalItems) {
                        this.applyMasonryLayout(masonryContainer);
                    }
                });
            } else {
                loadedCount++;
                if (loadedCount === totalItems) {
                    this.applyMasonryLayout(masonryContainer);
                }
            }
        });
    }
    
    applyMasonryLayout(container) {
        // Add staggered animation delays
        const items = container.querySelectorAll('.instagram-embed-container');
        items.forEach((item, index) => {
            item.style.animationDelay = `${index * 0.1}s`;
        });
    }
    
    initPinterest() {
        const pinterestContainer = this.element.querySelector('.instagram-pinterest');
        if (!pinterestContainer) return;
        
        // Pinterest-style grid with varying heights
        const items = pinterestContainer.querySelectorAll('.pinterest-item');
        
        items.forEach((item, index) => {
            // Add random height variations for Pinterest effect
            const heights = [35, 40, 45, 50, 55];
            const randomHeight = heights[index % heights.length];
            item.style.gridRowEnd = `span ${randomHeight}`;
            
            // Add staggered animations
            item.style.animationDelay = `${index * 0.15}s`;
        });
    }
    
    initTouchSupport() {
        let startX = 0;
        let startY = 0;
        let isDragging = false;
        
        this.carousel.addEventListener('touchstart', (e) => {
            startX = e.touches[0].clientX;
            startY = e.touches[0].clientY;
            isDragging = true;
            this.stopAutoplay();
        });
        
        this.carousel.addEventListener('touchmove', (e) => {
            if (!isDragging) return;
            e.preventDefault();
        });
        
        this.carousel.addEventListener('touchend', (e) => {
            if (!isDragging) return;
            
            const endX = e.changedTouches[0].clientX;
            const endY = e.changedTouches[0].clientY;
            const diffX = startX - endX;
            const diffY = startY - endY;
            
            // Check if horizontal swipe is more significant than vertical
            if (Math.abs(diffX) > Math.abs(diffY) && Math.abs(diffX) > 50) {
                if (diffX > 0) {
                    this.nextSlide();
                } else {
                    this.prevSlide();
                }
            }
            
            isDragging = false;
            if (this.settings.autoplay) {
                this.startAutoplay();
            }
        });
    }
    
    goToSlide(index) {
        if (index < 0 || index >= this.slides.length) return;
        
        // Hide all slides
        this.slides.forEach(slide => slide.classList.remove('active'));
        
        // Show target slide
        this.slides[index].classList.add('active');
        
        // Update indicators
        const indicators = this.carousel.querySelectorAll('.indicator');
        indicators.forEach((indicator, i) => {
            indicator.classList.toggle('active', i === index);
        });
        
        this.currentSlide = index;
    }
    
    nextSlide() {
        const nextIndex = (this.currentSlide + 1) % this.slides.length;
        this.goToSlide(nextIndex);
    }
    
    prevSlide() {
        const prevIndex = (this.currentSlide - 1 + this.slides.length) % this.slides.length;
        this.goToSlide(prevIndex);
    }
    
    nextMultiSlide() {
        if (this.currentPosition < this.maxPosition) {
            this.currentPosition = Math.min(this.currentPosition + this.scrollItems, this.maxPosition);
            this.updateMultiCarouselPosition();
        } else if (this.settings.autoplay) {
            // Loop back to beginning for autoplay
            this.currentPosition = 0;
            this.updateMultiCarouselPosition();
        }
    }
    
    prevMultiSlide() {
        if (this.currentPosition > 0) {
            this.currentPosition = Math.max(this.currentPosition - this.scrollItems, 0);
            this.updateMultiCarouselPosition();
        } else if (this.settings.autoplay) {
            // Loop to end for autoplay
            this.currentPosition = this.maxPosition;
            this.updateMultiCarouselPosition();
        }
    }
    
    updateMultiCarouselPosition() {
        if (!this.multiTrack) return;
        
        const itemWidth = this.multiItems[0].offsetWidth;
        const gap = 20; // Gap between items
        const translateX = -(this.currentPosition * (itemWidth + gap));
        
        this.multiTrack.style.transform = `translateX(${translateX}px)`;
    }
    
    initMultiTouchSupport() {
        let startX = 0;
        let startY = 0;
        let isDragging = false;
        
        this.multiCarousel.addEventListener('touchstart', (e) => {
            startX = e.touches[0].clientX;
            startY = e.touches[0].clientY;
            isDragging = true;
            this.stopAutoplay();
        });
        
        this.multiCarousel.addEventListener('touchmove', (e) => {
            if (!isDragging) return;
            e.preventDefault();
        });
        
        this.multiCarousel.addEventListener('touchend', (e) => {
            if (!isDragging) return;
            
            const endX = e.changedTouches[0].clientX;
            const endY = e.changedTouches[0].clientY;
            const diffX = startX - endX;
            const diffY = startY - endY;
            
            // Check if horizontal swipe is more significant than vertical
            if (Math.abs(diffX) > Math.abs(diffY) && Math.abs(diffX) > 50) {
                if (diffX > 0) {
                    this.nextMultiSlide();
                } else {
                    this.prevMultiSlide();
                }
            }
            
            isDragging = false;
            if (this.settings.autoplay) {
                this.startMultiAutoplay();
            }
        });
    }
    
    startAutoplay() {
        if (this.autoplayInterval) return;
        
        this.autoplayInterval = setInterval(() => {
            this.nextSlide();
        }, this.settings.speed || 5000);
    }
    
    startMultiAutoplay() {
        if (this.autoplayInterval) return;
        
        this.autoplayInterval = setInterval(() => {
            this.nextMultiSlide();
        }, this.settings.speed || 5000);
    }
    
    stopAutoplay() {
        if (this.autoplayInterval) {
            clearInterval(this.autoplayInterval);
            this.autoplayInterval = null;
        }
    }
    
    loadInstagramScript() {
        // Load Instagram embed script if not already loaded
        if (!document.querySelector('script[src*="instagram.com/embed.js"]')) {
            const script = document.createElement('script');
            script.src = '//www.instagram.com/embed.js';
            script.async = true;
            document.head.appendChild(script);
        } else {
            // If script already exists, process embeds
            if (window.instgrm) {
                window.instgrm.Embeds.process();
            }
        }
    }
    
    // Public method to refresh carousel
    refresh() {
        if (this.settings.layout === 'carousel') {
            this.currentSlide = 0;
            this.goToSlide(0);
        }
        this.loadInstagramScript();
    }
    
    // Cleanup method
    destroy() {
        this.stopAutoplay();
    }
}

// Initialize widgets when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    const widgets = document.querySelectorAll('.instagram-feed-widget');
    
    widgets.forEach(widget => {
        new InstagramFeedWidget(widget);
    });
});

// Re-initialize after Elementor preview updates
if (typeof elementorFrontend !== 'undefined') {
    elementorFrontend.hooks.addAction('frontend/element_ready/instagram_feed.default', function($scope) {
        const widget = $scope.find('.instagram-feed-widget')[0];
        if (widget) {
            new InstagramFeedWidget(widget);
        }
    });
}

// Expose class globally for manual initialization
window.InstagramFeedWidget = InstagramFeedWidget;
