<?php
/**
 * Instagram Feed Widget for Elementor
 * Displays Instagram images by username with unique styling
 *
 * @package RelayoutDesigns
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Make sure Elementor is active
if (!class_exists('\Elementor\Widget_Base')) {
    return;
}

class Instagram_Feed_Widget extends \Elementor\Widget_Base {

    public function get_name() {
        return 'instagram_feed';
    }

    public function get_title() {
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            return __('Instagram Feed', 'relayout-design') . ' <span style="color: #ff6b35; font-weight: bold; font-size: 10px; background: #fff3f0; padding: 2px 6px; border-radius: 3px; margin-left: 5px;">PRO</span>';
        }
        return __('Instagram Feed', 'relayout-design');
    }

    public function get_icon() {
        return 'eicon-instagram-post';
    }

    public function get_categories() {
        return ['relayout-designs'];
    }

    public function get_script_depends() {
        return ['instagram-feed-widget-script'];
    }

    public function get_style_depends() {
        return ['instagram-feed-widget-style'];
    }

    protected function register_controls() {
        // Content Section
        $this->start_controls_section(
            'content_section',
            [
                'label' => __('Instagram Settings', 'relayout-design'),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'instagram_info_notice',
            [
                'type' => \Elementor\Controls_Manager::RAW_HTML,
                'raw' => '<div style="background: #e8f5e8; padding: 15px; border-radius: 5px; border-left: 4px solid #4caf50;">
                    <h4 style="margin: 0 0 10px 0; color: #2e7d32;">📸 Instagram Embed Carousel</h4>
                    <p style="margin: 0; color: #555;"><strong>Native Instagram Embeds!</strong> Add Instagram post URLs and this widget will create a beautiful carousel with official Instagram embeds.</p>
                    <p style="margin: 10px 0 0 0; color: #666; font-size: 12px;"><em>Simply paste Instagram post URLs below and they will be displayed as native embeds in a carousel format.</em></p>
                </div>',
            ]
        );

        // Instagram Posts Repeater
        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'instagram_url',
            [
                'label' => __('Instagram Post URL', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::URL,
                'placeholder' => __('https://www.instagram.com/p/...', 'relayout-design'),
                'description' => __('Paste the Instagram post URL here', 'relayout-design'),
            ]
        );

        $this->add_control(
            'instagram_posts',
            [
                'label' => __('Instagram Posts', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'instagram_url' => [
                            'url' => 'https://www.instagram.com/p/example1/',
                        ],
                    ],
                    [
                        'instagram_url' => [
                            'url' => 'https://www.instagram.com/p/example2/',
                        ],
                    ],
                    [
                        'instagram_url' => [
                            'url' => 'https://www.instagram.com/p/example3/',
                        ],
                    ],
                ],
                'title_field' => '{{{ instagram_url.url }}}',
            ]
        );

        $this->add_control(
            'layout',
            [
                'label' => __('Layout', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'carousel',
                'options' => [
                    'carousel' => __('Carousel', 'relayout-design'),
                    'multi-carousel' => __('Multi-Item Carousel', 'relayout-design'),
                    'grid' => __('Grid', 'relayout-design'),
                    'masonry' => __('Masonry', 'relayout-design'),
                    'pinterest' => __('Pinterest Style', 'relayout-design'),
                    'single' => __('Single Post', 'relayout-design'),
                ],
            ]
        );

        $this->add_control(
            'columns',
            [
                'label' => __('Columns', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => '3',
                'options' => [
                    '1' => __('1 Column', 'relayout-design'),
                    '2' => __('2 Columns', 'relayout-design'),
                    '3' => __('3 Columns', 'relayout-design'),
                    '4' => __('4 Columns', 'relayout-design'),
                    '5' => __('5 Columns', 'relayout-design'),
                ],
                'condition' => [
                    'layout!' => ['carousel', 'single'],
                ],
            ]
        );

        $this->add_control(
            'multi_carousel_items',
            [
                'label' => __('Items to Show', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => '3',
                'options' => [
                    '2' => __('2 Items', 'relayout-design'),
                    '3' => __('3 Items', 'relayout-design'),
                    '4' => __('4 Items', 'relayout-design'),
                    '5' => __('5 Items', 'relayout-design'),
                ],
                'condition' => [
                    'layout' => 'multi-carousel',
                ],
            ]
        );

        $this->add_control(
            'multi_carousel_scroll',
            [
                'label' => __('Scroll Items', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => '1',
                'options' => [
                    '1' => __('1 Item', 'relayout-design'),
                    '2' => __('2 Items', 'relayout-design'),
                    '3' => __('3 Items', 'relayout-design'),
                ],
                'condition' => [
                    'layout' => 'multi-carousel',
                ],
                'description' => __('Number of items to scroll at once', 'relayout-design'),
            ]
        );

        $this->add_control(
            'gap',
            [
                'label' => __('Gap', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 15,
                ],
                'selectors' => [
                    '{{WRAPPER}} .instagram-grid' => 'gap: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .instagram-masonry' => 'gap: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .instagram-pinterest' => 'gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'carousel_autoplay',
            [
                'label' => __('Auto Play', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
                'condition' => [
                    'layout' => ['carousel', 'multi-carousel'],
                ],
                'description' => __('Automatically slide through posts', 'relayout-design'),
            ]
        );

        $this->add_control(
            'carousel_speed',
            [
                'label' => __('Auto Play Speed (seconds)', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 5,
                'min' => 2,
                'max' => 10,
                'condition' => [
                    'layout' => ['carousel', 'multi-carousel'],
                    'carousel_autoplay' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - General
        $this->start_controls_section(
            'style_general_section',
            [
                'label' => __('General Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'item_spacing',
            [
                'label' => __('Item Spacing', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 15,
                ],
                'selectors' => [
                    '{{WRAPPER}} .instagram-grid' => 'gap: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .instagram-carousel .swiper-slide' => 'padding-right: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'item_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 10,
                    'right' => 10,
                    'bottom' => 10,
                    'left' => 10,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .instagram-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'item_shadow',
                'selector' => '{{WRAPPER}} .instagram-item',
            ]
        );

        $this->end_controls_section();

        // Carousel Navigation Style Section
        $this->start_controls_section(
            'carousel_navigation_section',
            [
                'label' => __('Carousel Navigation', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'layout' => ['carousel', 'multi-carousel'],
                ],
            ]
        );

        $this->add_control(
            'show_navigation',
            [
                'label' => __('Show Navigation', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
                'description' => __('Show previous/next navigation arrows', 'relayout-design'),
            ]
        );

        $this->add_control(
            'navigation_prev_icon',
            [
                'label' => __('Previous Icon', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-chevron-left',
                    'library' => 'fa-solid',
                ],
                'condition' => [
                    'show_navigation' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'navigation_next_icon',
            [
                'label' => __('Next Icon', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-chevron-right',
                    'library' => 'fa-solid',
                ],
                'condition' => [
                    'show_navigation' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'navigation_size',
            [
                'label' => __('Icon Size', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 12,
                        'max' => 40,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 18,
                ],
                'selectors' => [
                    '{{WRAPPER}} .carousel-nav' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .carousel-nav svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'show_navigation' => 'yes',
                ],
            ]
        );

        $this->start_controls_tabs('navigation_style_tabs');

        $this->start_controls_tab(
            'navigation_normal_tab',
            [
                'label' => __('Normal', 'relayout-design'),
                'condition' => [
                    'show_navigation' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'navigation_color',
            [
                'label' => __('Icon Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .carousel-nav' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .carousel-nav svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'navigation_background',
            [
                'label' => __('Background Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => 'rgba(0,0,0,0.5)',
                'selectors' => [
                    '{{WRAPPER}} .carousel-nav' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'navigation_hover_tab',
            [
                'label' => __('Hover', 'relayout-design'),
                'condition' => [
                    'show_navigation' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'navigation_hover_color',
            [
                'label' => __('Icon Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .carousel-nav:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .carousel-nav:hover svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'navigation_hover_background',
            [
                'label' => __('Background Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => 'rgba(0,0,0,0.8)',
                'selectors' => [
                    '{{WRAPPER}} .carousel-nav:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_responsive_control(
            'navigation_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 50,
                    'right' => 50,
                    'bottom' => 50,
                    'left' => 50,
                    'unit' => '%',
                ],
                'selectors' => [
                    '{{WRAPPER}} .carousel-nav' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'show_navigation' => 'yes',
                ],
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'navigation_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'default' => [
                    'top' => 12,
                    'right' => 12,
                    'bottom' => 12,
                    'left' => 12,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .carousel-nav' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'show_navigation' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'navigation_position',
            [
                'label' => __('Position', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'center',
                'options' => [
                    'top' => __('Top', 'relayout-design'),
                    'center' => __('Center', 'relayout-design'),
                    'bottom' => __('Bottom', 'relayout-design'),
                ],
                'condition' => [
                    'show_navigation' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Overlay
        $this->start_controls_section(
            'style_overlay_section',
            [
                'label' => __('Overlay Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'overlay_background',
            [
                'label' => __('Overlay Background', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => 'rgba(0, 0, 0, 0.7)',
                'selectors' => [
                    '{{WRAPPER}} .instagram-overlay' => 'background: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'overlay_text_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .instagram-overlay' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'caption_typography',
                'selector' => '{{WRAPPER}} .instagram-caption',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'stats_typography',
                'selector' => '{{WRAPPER}} .instagram-stats',
            ]
        );

        $this->end_controls_section();

        // Style Section - Loading
        $this->start_controls_section(
            'style_loading_section',
            [
                'label' => __('Loading Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'loading_color',
            [
                'label' => __('Loading Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#3498db',
                'selectors' => [
                    '{{WRAPPER}} .instagram-loading' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .loading-spinner' => 'border-top-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        // Check if premium features are available
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            $upgrade_prompt = relayout_designs_show_upgrade_prompt('Instagram Feed Widget');
            if ($upgrade_prompt) {
                echo $upgrade_prompt;
                return;
            }
        }
        
        $settings = $this->get_settings_for_display();
        
        if (empty($settings['instagram_posts'])) {
            echo '<div class="instagram-error">Please add Instagram post URLs in the widget settings.</div>';
            return;
        }

        $widget_settings = [
            'layout' => $settings['layout'],
            'columns' => $settings['columns'],
            'multiCarouselItems' => $settings['multi_carousel_items'] ?? '3',
            'multiCarouselScroll' => $settings['multi_carousel_scroll'] ?? '1',
            'autoplay' => $settings['carousel_autoplay'] === 'yes',
            'speed' => intval($settings['carousel_speed']) * 1000, // Convert to milliseconds
            'showNavigation' => $settings['show_navigation'] === 'yes',
            'navigationPrevIcon' => $settings['navigation_prev_icon'],
            'navigationNextIcon' => $settings['navigation_next_icon'],
            'navigationPosition' => $settings['navigation_position'] ?? 'center',
        ];
        ?>
        
        <div class="instagram-feed-widget" data-settings="<?php echo esc_attr(json_encode($widget_settings)); ?>">
            <?php if ($settings['layout'] === 'carousel'): ?>
                <div class="instagram-carousel">
                    <div class="carousel-container">
                        <div class="carousel-track">
                            <?php foreach ($settings['instagram_posts'] as $index => $post): ?>
                                <?php if (!empty($post['instagram_url']['url'])): ?>
                                    <div class="carousel-slide <?php echo $index === 0 ? 'active' : ''; ?>">
                                        <div class="instagram-embed-container">
                                            <?php echo $this->generateInstagramEmbed($post['instagram_url']['url']); ?>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    
                    <?php if (count($settings['instagram_posts']) > 1 && $settings['show_navigation'] === 'yes'): ?>
                        <div class="carousel-controls nav-position-<?php echo esc_attr($settings['navigation_position']); ?>">
                            <button class="carousel-nav carousel-prev" aria-label="Previous post">
                                <?php \Elementor\Icons_Manager::render_icon($settings['navigation_prev_icon'], ['aria-hidden' => 'true']); ?>
                            </button>
                            <button class="carousel-nav carousel-next" aria-label="Next post">
                                <?php \Elementor\Icons_Manager::render_icon($settings['navigation_next_icon'], ['aria-hidden' => 'true']); ?>
                            </button>
                        </div>
                        
                        <div class="carousel-indicators">
                            <?php foreach ($settings['instagram_posts'] as $index => $post): ?>
                                <?php if (!empty($post['instagram_url']['url'])): ?>
                                    <button class="indicator <?php echo $index === 0 ? 'active' : ''; ?>" data-slide="<?php echo $index; ?>"></button>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
                
            <?php elseif ($settings['layout'] === 'multi-carousel'): ?>
                <div class="instagram-multi-carousel" data-items="<?php echo esc_attr($settings['multi_carousel_items']); ?>" data-scroll="<?php echo esc_attr($settings['multi_carousel_scroll']); ?>">
                    <div class="multi-carousel-container">
                        <div class="multi-carousel-track">
                            <?php foreach ($settings['instagram_posts'] as $index => $post): ?>
                                <?php if (!empty($post['instagram_url']['url'])): ?>
                                    <div class="multi-carousel-item">
                                        <div class="instagram-embed-container">
                                            <?php echo $this->generateInstagramEmbed($post['instagram_url']['url']); ?>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    
                    <?php if (count($settings['instagram_posts']) > intval($settings['multi_carousel_items']) && $settings['show_navigation'] === 'yes'): ?>
                        <div class="carousel-controls nav-position-<?php echo esc_attr($settings['navigation_position']); ?>">
                            <button class="carousel-nav carousel-prev" aria-label="Previous posts">
                                <?php \Elementor\Icons_Manager::render_icon($settings['navigation_prev_icon'], ['aria-hidden' => 'true']); ?>
                            </button>
                            <button class="carousel-nav carousel-next" aria-label="Next posts">
                                <?php \Elementor\Icons_Manager::render_icon($settings['navigation_next_icon'], ['aria-hidden' => 'true']); ?>
                            </button>
                        </div>
                    <?php endif; ?>
                </div>
                
            <?php elseif ($settings['layout'] === 'grid'): ?>
                <div class="instagram-grid instagram-columns-<?php echo esc_attr($settings['columns']); ?>">
                    <?php foreach ($settings['instagram_posts'] as $post): ?>
                        <?php if (!empty($post['instagram_url']['url'])): ?>
                            <div class="instagram-embed-container">
                                <?php echo $this->generateInstagramEmbed($post['instagram_url']['url']); ?>
                            </div>
                        <?php endif; ?>
                    <?php endforeach; ?>
                </div>
                
            <?php elseif ($settings['layout'] === 'masonry'): ?>
                <div class="instagram-masonry instagram-columns-<?php echo esc_attr($settings['columns']); ?>">
                    <?php foreach ($settings['instagram_posts'] as $post): ?>
                        <?php if (!empty($post['instagram_url']['url'])): ?>
                            <div class="instagram-embed-container">
                                <?php echo $this->generateInstagramEmbed($post['instagram_url']['url']); ?>
                            </div>
                        <?php endif; ?>
                    <?php endforeach; ?>
                </div>
                
            <?php elseif ($settings['layout'] === 'pinterest'): ?>
                <div class="instagram-pinterest instagram-columns-<?php echo esc_attr($settings['columns']); ?>">
                    <?php foreach ($settings['instagram_posts'] as $post): ?>
                        <?php if (!empty($post['instagram_url']['url'])): ?>
                            <div class="instagram-embed-container pinterest-item">
                                <?php echo $this->generateInstagramEmbed($post['instagram_url']['url']); ?>
                            </div>
                        <?php endif; ?>
                    <?php endforeach; ?>
                </div>
                
            <?php else: ?>
                <div class="instagram-single">
                    <?php 
                    $firstPost = !empty($settings['instagram_posts'][0]['instagram_url']['url']) ? $settings['instagram_posts'][0]['instagram_url']['url'] : '';
                    if ($firstPost): ?>
                        <div class="instagram-embed-container">
                            <?php echo $this->generateInstagramEmbed($firstPost); ?>
                        </div>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
        
        <?php
    }

    private function generateInstagramEmbed($url) {
        // Extract post ID from URL
        preg_match('/\/p\/([^\/\?]+)/', $url, $matches);
        if (empty($matches[1])) {
            preg_match('/\/reel\/([^\/\?]+)/', $url, $matches);
        }
        
        if (empty($matches[1])) {
            return '<div class="instagram-error">Invalid Instagram URL format</div>';
        }
        
        $postId = $matches[1];
        $embedUrl = "https://www.instagram.com/p/{$postId}/?utm_source=ig_embed&utm_campaign=loading";
        
        return '<blockquote class="instagram-media" data-instgrm-captioned data-instgrm-permalink="' . esc_url($embedUrl) . '" data-instgrm-version="14" style="background:#FFF; border:0; border-radius:3px; box-shadow:0 0 1px 0 rgba(0,0,0,0.5),0 1px 10px 0 rgba(0,0,0,0.15); margin: 1px; max-width:540px; min-width:326px; padding:0; width:99.375%; width:-webkit-calc(100% - 2px); width:calc(100% - 2px);">
            <div style="padding:16px;">
                <a href="' . esc_url($embedUrl) . '" style="background:#FFFFFF; line-height:0; padding:0 0; text-align:center; text-decoration:none; width:100%;" target="_blank">
                    <div style="display: flex; flex-direction: row; align-items: center;">
                        <div style="background-color: #F4F4F4; border-radius: 50%; flex-grow: 0; height: 40px; margin-right: 14px; width: 40px;"></div>
                        <div style="display: flex; flex-direction: column; flex-grow: 1; justify-content: center;">
                            <div style="background-color: #F4F4F4; border-radius: 4px; flex-grow: 0; height: 14px; margin-bottom: 6px; width: 100px;"></div>
                            <div style="background-color: #F4F4F4; border-radius: 4px; flex-grow: 0; height: 14px; width: 60px;"></div>
                        </div>
                    </div>
                    <div style="padding: 19% 0;"></div>
                    <div style="display:block; height:50px; margin:0 auto 12px; width:50px;">
                        <svg width="50px" height="50px" viewBox="0 0 60 60" version="1.1" xmlns="https://www.w3.org/2000/svg">
                            <g stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
                                <g transform="translate(-511.000000, -20.000000)" fill="#000000">
                                    <g><path d="M556.869,30.41 C554.814,30.41 553.148,32.076 553.148,34.131 C553.148,36.186 554.814,37.852 556.869,37.852 C558.924,37.852 560.59,36.186 560.59,34.131 C560.59,32.076 558.924,30.41 556.869,30.41 M541,60.657 C535.114,60.657 530.342,55.887 530.342,50 C530.342,44.114 535.114,39.342 541,39.342 C546.887,39.342 551.658,44.114 551.658,50 C551.658,55.887 546.887,60.657 541,60.657 M541,33.886 C532.1,33.886 524.886,41.1 524.886,50 C524.886,58.899 532.1,66.113 541,66.113 C549.9,66.113 557.115,58.899 557.115,50 C557.115,41.1 549.9,33.886 541,33.886 M565.378,62.101 C565.244,65.022 564.756,66.606 564.346,67.663 C563.803,69.06 563.154,70.057 562.106,71.106 C561.058,72.155 560.06,72.803 558.662,73.347 C557.607,73.757 556.021,74.244 553.102,74.378 C549.944,74.521 548.997,74.552 541,74.552 C533.003,74.552 532.056,74.521 528.898,74.378 C525.979,74.244 524.393,73.757 523.338,73.347 C521.94,72.803 520.942,72.155 519.894,71.106 C518.846,70.057 518.197,69.06 517.654,67.663 C517.244,66.606 516.755,65.022 516.623,62.101 C516.479,58.943 516.448,57.996 516.448,50 C516.448,42.003 516.479,41.056 516.623,37.899 C516.755,34.978 517.244,33.391 517.654,32.338 C518.197,30.938 518.846,29.942 519.894,28.894 C520.942,27.846 521.94,27.196 523.338,26.654 C524.393,26.244 525.979,25.756 528.898,25.623 C532.057,25.479 533.004,25.448 541,25.448 C548.997,25.448 549.943,25.479 553.102,25.623 C556.021,25.756 557.607,26.244 558.662,26.654 C560.06,27.196 561.058,27.846 562.106,28.894 C563.154,29.942 563.803,30.938 564.346,32.338 C564.756,33.391 565.244,34.978 565.378,37.899 C565.522,41.056 565.552,42.003 565.552,50 C565.552,57.996 565.522,58.943 565.378,62.101 M570.82,37.631 C570.674,34.438 570.167,32.258 569.425,30.349 C568.659,28.377 567.633,26.702 565.965,25.035 C564.297,23.368 562.623,22.342 560.652,21.575 C558.743,20.834 556.562,20.326 553.369,20.18 C550.169,20.033 549.148,20 541,20 C532.853,20 531.831,20.033 528.631,20.18 C525.438,20.326 523.257,20.834 521.349,21.575 C519.376,22.342 517.703,23.368 516.035,25.035 C514.368,26.702 513.342,28.377 512.574,30.349 C511.834,32.258 511.326,34.438 511.181,37.631 C511.035,40.831 511,41.851 511,50 C511,58.147 511.035,59.17 511.181,62.369 C511.326,65.562 511.834,67.743 512.574,69.651 C513.342,71.625 514.368,73.296 516.035,74.965 C517.703,76.634 519.376,77.658 521.349,78.425 C523.257,79.167 525.438,79.673 528.631,79.82 C531.831,79.965 532.853,80.001 541,80.001 C549.148,80.001 550.169,79.965 553.369,79.82 C556.562,79.673 558.743,79.167 560.652,78.425 C562.623,77.658 564.297,76.634 565.965,74.965 C567.633,73.296 568.659,71.625 569.425,69.651 C570.167,67.743 570.674,65.562 570.82,62.369 C570.966,59.17 571,58.147 571,50 C571,41.851 570.966,40.831 570.82,37.631"></path></g>
                                </g>
                            </g>
                        </svg>
                    </div>
                    <div style="padding-top: 8px;">
                        <div style="color:#3897f0; font-family:Arial,sans-serif; font-size:14px; font-style:normal; font-weight:550; line-height:18px;">View this post on Instagram</div>
                    </div>
                    <div style="padding: 12.5% 0;"></div>
                </a>
            </div>
        </blockquote>
        <script async src="//www.instagram.com/embed.js"></script>';
    }
}
