/**
 * Reading Progress Bar Widget JavaScript
 * Handles scroll progress calculation and progress bar updates
 */

class ReadingProgressBarWidget {
    constructor(element) {
        this.element = element;
        this.settings = JSON.parse(element.dataset.settings || '{}');
        this.progressBar = element.querySelector('.reading-progress-bar');
        this.progressFill = element.querySelector('.reading-progress-fill');
        this.percentageText = element.querySelector('.reading-progress-percentage');
        this.isFixed = false;
        this.currentProgress = 0;
        this.animationFrame = null;
        
        this.init();
    }

    init() {
        this.setupProgressBar();
        this.bindEvents();
        this.updateProgress();
    }

    setupProgressBar() {
        if (!this.progressBar) return;
        
        // Make progress bar fixed positioned
        this.progressBar.style.position = 'fixed';
        this.progressBar.style.left = '0';
        this.progressBar.style.width = '100%';
        this.progressBar.style.zIndex = this.settings.zIndex || 9999;
        
        // Set position (top or bottom)
        if (this.settings.position === 'bottom') {
            this.progressBar.style.bottom = '0';
            this.progressBar.style.top = 'auto';
        } else {
            this.progressBar.style.top = '0';
            this.progressBar.style.bottom = 'auto';
        }
        
        this.isFixed = true;
    }

    bindEvents() {
        // Throttled scroll event
        let scrollTimeout = null;
        window.addEventListener('scroll', () => {
            if (scrollTimeout) {
                cancelAnimationFrame(scrollTimeout);
            }
            scrollTimeout = requestAnimationFrame(() => this.updateProgress());
        });

        // Window resize event
        window.addEventListener('resize', () => {
            this.updateProgress();
        });

        // Update on page load
        window.addEventListener('load', () => {
            setTimeout(() => this.updateProgress(), 100);
        });
    }

    calculateProgress() {
        const windowHeight = window.innerHeight;
        const documentHeight = Math.max(
            document.body.scrollHeight,
            document.body.offsetHeight,
            document.documentElement.clientHeight,
            document.documentElement.scrollHeight,
            document.documentElement.offsetHeight
        );
        
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
        const scrollableDistance = documentHeight - windowHeight;
        
        if (scrollableDistance <= 0) {
            return 0;
        }
        
        const progress = Math.min(Math.max(scrollTop / scrollableDistance, 0), 1);
        return Math.round(progress * 100);
    }

    updateProgress() {
        if (!this.progressFill) return;
        
        const progress = this.calculateProgress();
        
        // Only update if progress changed significantly
        if (Math.abs(progress - this.currentProgress) < 1) {
            return;
        }
        
        this.currentProgress = progress;
        
        // Apply animation based on style
        this.applyAnimation(progress);
        
        // Update percentage text
        if (this.percentageText && this.settings.showPercentage) {
            this.percentageText.textContent = progress + '%';
        }
        
        // Trigger custom event
        this.element.dispatchEvent(new CustomEvent('progressUpdate', {
            detail: { progress: progress }
        }));
    }

    applyAnimation(progress) {
        const fillWidth = progress + '%';
        
        switch (this.settings.animationStyle) {
            case 'stepped':
                // Step animation with CSS steps
                this.progressFill.style.transition = 'width 0.1s steps(10, end)';
                break;
                
            case 'bounce':
                // Bounce animation
                this.progressFill.style.transition = 'width 0.6s cubic-bezier(0.68, -0.55, 0.265, 1.55)';
                break;
                
            case 'elastic':
                // Elastic animation
                this.progressFill.style.transition = 'width 0.8s cubic-bezier(0.175, 0.885, 0.32, 1.275)';
                break;
                
            case 'smooth':
            default:
                // Smooth animation
                this.progressFill.style.transition = 'width 0.3s ease-out';
                break;
        }
        
        this.progressFill.style.width = fillWidth;
    }

    destroy() {
        // Clean up event listeners
        window.removeEventListener('scroll', this.updateProgress);
        window.removeEventListener('resize', this.updateProgress);
        window.removeEventListener('load', this.updateProgress);
        
        if (this.animationFrame) {
            cancelAnimationFrame(this.animationFrame);
        }
    }
}

// Initialize widgets when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    initializeReadingProgressBars();
});

// Also initialize on Elementor frontend
if (typeof window.elementorFrontend !== 'undefined') {
    window.elementorFrontend.hooks.addAction('frontend/element_ready/reading_progress_bar.default', function($scope) {
        initializeReadingProgressBars($scope[0]);
    });
}

// Initialize function
function initializeReadingProgressBars(container = document) {
    const widgets = container.querySelectorAll('.reading-progress-widget');
    
    widgets.forEach(widget => {
        // Avoid double initialization
        if (widget.readingProgressInstance) {
            widget.readingProgressInstance.destroy();
        }
        
        widget.readingProgressInstance = new ReadingProgressBarWidget(widget);
    });
}

// Handle SPA navigation and dynamic content
if (typeof MutationObserver !== 'undefined') {
    const observer = new MutationObserver(function(mutations) {
        let shouldReinit = false;
        
        mutations.forEach(function(mutation) {
            if (mutation.type === 'childList') {
                mutation.addedNodes.forEach(function(node) {
                    if (node.nodeType === 1) { // Element node
                        if (node.matches && node.matches('.reading-progress-widget')) {
                            shouldReinit = true;
                        } else if (node.querySelector && node.querySelector('.reading-progress-widget')) {
                            shouldReinit = true;
                        }
                    }
                });
            }
        });
        
        if (shouldReinit) {
            setTimeout(initializeReadingProgressBars, 100);
        }
    });
    
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
}

// Expose for external use
window.ReadingProgressBarWidget = ReadingProgressBarWidget;
window.initializeReadingProgressBars = initializeReadingProgressBars;
