<?php
/**
 * Reading Progress Bar Widget for Elementor
 * Displays a progress bar showing scroll progress through page content
 *
 * @package RelayoutDesigns
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Make sure Elementor is active
if (!class_exists('\Elementor\Widget_Base')) {
    return;
}

class Reading_Progress_Bar_Widget extends \Elementor\Widget_Base {

    public function get_name() {
        return 'reading_progress_bar';
    }

    public function get_title() {
        return __('Reading Progress Bar', 'relayout-design');
    }

    public function get_icon() {
        return 'eicon-progress-tracker';
    }

    public function get_categories() {
        return ['relayout-designs'];
    }

    public function get_script_depends() {
        return ['reading-progress-bar-widget-script'];
    }

    public function get_style_depends() {
        return ['reading-progress-bar-widget-style'];
    }

    protected function register_controls() {
        // Content Section
        $this->start_controls_section(
            'content_section',
            [
                'label' => __('Progress Bar Settings', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'position',
            [
                'label' => __('Position', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'top',
                'options' => [
                    'top' => __('Top of Page', 'relayout-design'),
                    'bottom' => __('Bottom of Page', 'relayout-design'),
                ],
            ]
        );

        $this->add_control(
            'show_percentage',
            [
                'label' => __('Show Percentage', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'no',
                'description' => __('Display percentage text on the progress bar', 'relayout-design'),
            ]
        );

        $this->add_control(
            'percentage_position',
            [
                'label' => __('Percentage Position', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'center',
                'options' => [
                    'left' => __('Left', 'relayout-design'),
                    'center' => __('Center', 'relayout-design'),
                    'right' => __('Right', 'relayout-design'),
                ],
                'condition' => [
                    'show_percentage' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'animation_style',
            [
                'label' => __('Animation Style', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'smooth',
                'options' => [
                    'smooth' => __('Smooth', 'relayout-design'),
                    'stepped' => __('Stepped', 'relayout-design'),
                    'bounce' => __('Bounce', 'relayout-design'),
                    'elastic' => __('Elastic', 'relayout-design'),
                ],
            ]
        );

        $this->add_control(
            'z_index',
            [
                'label' => __('Z-Index', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 9999,
                'min' => 1,
                'max' => 99999,
                'description' => __('Controls layering order (higher values appear on top)', 'relayout-design'),
            ]
        );

        $this->end_controls_section();

        // Style Section - Progress Bar
        $this->start_controls_section(
            'progress_bar_style_section',
            [
                'label' => __('Progress Bar Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'thickness',
            [
                'label' => __('Thickness', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 2,
                        'max' => 20,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 4,
                ],
                'selectors' => [
                    '{{WRAPPER}} .reading-progress-bar' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'background_color',
            [
                'label' => __('Background Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => 'rgba(0,0,0,0.1)',
                'selectors' => [
                    '{{WRAPPER}} .reading-progress-bar' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'fill_background',
                'label' => __('Fill Background', 'relayout-design'),
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .reading-progress-fill',
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                    'color' => [
                        'default' => '#007cba',
                    ],
                ],
            ]
        );

        $this->add_control(
            'border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 0,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .reading-progress-bar' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .reading-progress-fill' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'box_shadow',
                'label' => __('Box Shadow', 'relayout-design'),
                'selector' => '{{WRAPPER}} .reading-progress-bar',
            ]
        );

        $this->end_controls_section();

        // Style Section - Percentage Text
        $this->start_controls_section(
            'percentage_style_section',
            [
                'label' => __('Percentage Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_percentage' => 'yes',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'percentage_typography',
                'label' => __('Typography', 'relayout-design'),
                'selector' => '{{WRAPPER}} .reading-progress-percentage',
            ]
        );

        $this->add_control(
            'percentage_color',
            [
                'label' => __('Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .reading-progress-percentage' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'percentage_background',
            [
                'label' => __('Background Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => 'rgba(0,0,0,0.7)',
                'selectors' => [
                    '{{WRAPPER}} .reading-progress-percentage' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'percentage_padding',
            [
                'label' => __('Padding', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'default' => [
                    'top' => 4,
                    'right' => 8,
                    'bottom' => 4,
                    'left' => 8,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .reading-progress-percentage' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'percentage_border_radius',
            [
                'label' => __('Border Radius', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 4,
                    'right' => 4,
                    'bottom' => 4,
                    'left' => 4,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .reading-progress-percentage' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        
        $widget_settings = [
            'position' => $settings['position'],
            'showPercentage' => $settings['show_percentage'] === 'yes',
            'percentagePosition' => $settings['percentage_position'] ?? 'center',
            'animationStyle' => $settings['animation_style'],
            'zIndex' => intval($settings['z_index']),
        ];
        ?>
        
        <div class="reading-progress-widget" data-settings="<?php echo esc_attr(json_encode($widget_settings)); ?>">
            <div class="reading-progress-bar position-<?php echo esc_attr($settings['position']); ?> animation-<?php echo esc_attr($settings['animation_style']); ?>" style="z-index: <?php echo esc_attr($settings['z_index']); ?>;">
                <div class="reading-progress-fill"></div>
                <?php if ($settings['show_percentage'] === 'yes') : ?>
                    <div class="reading-progress-percentage position-<?php echo esc_attr($settings['percentage_position']); ?>">0%</div>
                <?php endif; ?>
            </div>
        </div>
        
        <?php
    }

    protected function content_template() {
        ?>
        <#
        var showPercentage = settings.show_percentage === 'yes';
        var percentagePosition = settings.percentage_position || 'center';
        var animationStyle = settings.animation_style || 'smooth';
        var position = settings.position || 'top';
        var zIndex = settings.z_index || 9999;
        #>
        
        <div class="reading-progress-widget">
            <div class="reading-progress-bar position-{{ position }} animation-{{ animationStyle }}" style="z-index: {{ zIndex }};">
                <div class="reading-progress-fill" style="width: 45%;"></div>
                <# if (showPercentage) { #>
                    <div class="reading-progress-percentage position-{{ percentagePosition }}">45%</div>
                <# } #>
            </div>
        </div>
        
        <?php
    }
}
