/**
 * Weather Widget JavaScript
 * Handles wttr.in API calls and real-time weather updates (No API key required!)
 */

class WeatherWidget {
    constructor(element) {
        this.element = element;
        this.config = JSON.parse(element.dataset.config || '{}');
        this.cache = new Map();
        this.cacheExpiry = 30 * 60 * 1000; // 30 minutes
        
        this.loadingEl = element.querySelector('.weather-loading');
        this.errorEl = element.querySelector('.weather-error');
        this.contentEl = element.querySelector('.weather-content');
        this.lastUpdatedEl = element.querySelector('.update-time');
        
        this.init();
    }

    init() {
        if (!this.config.city_name) {
            this.showError('City name is required. Please configure your location.');
            return;
        }

        this.loadWeatherData();
        
        // Set up auto-refresh
        const interval = (this.config.update_interval || 30) * 60 * 1000;
        setInterval(() => {
            this.loadWeatherData();
        }, interval);
    }

    async loadWeatherData() {
        try {
            this.showLoading();
            
            const cacheKey = `weather_${this.config.city_name}_${this.config.units}`;
            if (this.cache.has(cacheKey)) {
                const cached = this.cache.get(cacheKey);
                if (Date.now() - cached.timestamp < this.cacheExpiry) {
                    this.displayWeather(cached.data);
                    this.updateLastUpdated();
                    this.showContent();
                    return;
                }
            }

            const weatherData = await this.fetchWeatherData();
            
            this.cache.set(cacheKey, {
                data: weatherData,
                timestamp: Date.now()
            });

            this.displayWeather(weatherData);
            this.updateLastUpdated();
            this.showContent();
            
        } catch (error) {
            console.error('Weather widget error:', error);
            this.showError(error.message || 'Failed to load weather data');
        }
    }

    async fetchWeatherData() {
        const city = encodeURIComponent(this.config.city_name);
        const units = this.config.units === 'imperial' ? 'u' : 'm'; // u = imperial, m = metric
        
        // Use wttr.in API - no API key required!
        const url = `https://wttr.in/${city}?format=j1&${units}`;
        
        const response = await fetch(url);
        if (!response.ok) {
            throw new Error(`Weather service error: ${response.status}`);
        }

        const data = await response.json();
        
        if (!data.current_condition || !data.current_condition[0]) {
            throw new Error('Invalid weather data received');
        }

        return this.parseWeatherData(data);
    }

    parseWeatherData(data) {
        const current = data.current_condition[0];
        const location = data.nearest_area[0];
        const weather = data.weather || [];

        // Map weather codes to emojis
        const weatherEmojis = {
            '113': '☀️', // Sunny
            '116': '⛅', // Partly cloudy
            '119': '☁️', // Cloudy
            '122': '☁️', // Overcast
            '143': '🌫️', // Mist
            '176': '🌦️', // Patchy rain possible
            '179': '🌨️', // Patchy snow possible
            '182': '🌧️', // Patchy sleet possible
            '185': '🌧️', // Patchy freezing drizzle possible
            '200': '⛈️', // Thundery outbreaks possible
            '227': '🌨️', // Blowing snow
            '230': '❄️', // Blizzard
            '248': '🌫️', // Fog
            '260': '🌫️', // Freezing fog
            '263': '🌦️', // Patchy light drizzle
            '266': '🌧️', // Light drizzle
            '281': '🌧️', // Freezing drizzle
            '284': '🌧️', // Heavy freezing drizzle
            '293': '🌦️', // Patchy light rain
            '296': '🌧️', // Light rain
            '299': '🌧️', // Moderate rain at times
            '302': '🌧️', // Moderate rain
            '305': '🌧️', // Heavy rain at times
            '308': '🌧️', // Heavy rain
            '311': '🌧️', // Light freezing rain
            '314': '🌧️', // Moderate or heavy freezing rain
            '317': '🌧️', // Light sleet
            '320': '🌧️', // Moderate or heavy sleet
            '323': '🌨️', // Patchy light snow
            '326': '🌨️', // Light snow
            '329': '🌨️', // Patchy moderate snow
            '332': '🌨️', // Moderate snow
            '335': '🌨️', // Patchy heavy snow
            '338': '❄️', // Heavy snow
            '350': '🌧️', // Ice pellets
            '353': '🌦️', // Light rain shower
            '356': '🌧️', // Moderate or heavy rain shower
            '359': '🌧️', // Torrential rain shower
            '362': '🌧️', // Light sleet showers
            '365': '🌧️', // Moderate or heavy sleet showers
            '368': '🌨️', // Light snow showers
            '371': '❄️', // Moderate or heavy snow showers
            '374': '🌧️', // Light showers of ice pellets
            '377': '🌧️', // Moderate or heavy showers of ice pellets
            '386': '⛈️', // Patchy light rain with thunder
            '389': '⛈️', // Moderate or heavy rain with thunder
            '392': '⛈️', // Patchy light snow with thunder
            '395': '⛈️'  // Moderate or heavy snow with thunder
        };

        return {
            current: {
                temp: parseInt(current.temp_C || current.temp_F),
                feels_like: parseInt(current.FeelsLikeC || current.FeelsLikeF),
                humidity: parseInt(current.humidity),
                wind_speed: parseInt(current.windspeedKmph),
                visibility: parseInt(current.visibility),
                description: current.weatherDesc[0].value,
                emoji: weatherEmojis[current.weatherCode] || '🌤️',
                code: current.weatherCode
            },
            location: {
                name: this.config.city_name.split(',')[0].trim(), // Use user-selected city name
                country: this.config.city_name.includes(',') ? this.config.city_name.split(',')[1].trim() : location.country[0].value
            },
            forecast: weather.slice(0, 3).map(day => ({
                date: day.date,
                max_temp: parseInt(day.maxtempC || day.maxtempF),
                min_temp: parseInt(day.mintempC || day.mintempF),
                description: day.hourly[0].weatherDesc[0].value,
                emoji: weatherEmojis[day.hourly[0].weatherCode] || '🌤️'
            }))
        };
    }

    displayWeather(weatherData) {
        // Update current weather
        const tempEl = this.contentEl.querySelector('.current-temp');
        const descEl = this.contentEl.querySelector('.weather-description');
        const locationEl = this.contentEl.querySelector('.location');
        const emojiEl = this.contentEl.querySelector('.weather-emoji');

        const unitSymbol = this.getUnitSymbol();
        
        tempEl.textContent = `${weatherData.current.temp}${unitSymbol}`;
        descEl.textContent = weatherData.current.description;
        locationEl.textContent = `${weatherData.location.name}, ${weatherData.location.country}`;
        emojiEl.textContent = weatherData.current.emoji;

        // Set weather condition for background styling
        this.element.setAttribute('data-condition', this.getConditionFromCode(weatherData.current.code));

        // Update weather details
        const feelsLikeEl = this.contentEl.querySelector('.feels-like');
        const humidityEl = this.contentEl.querySelector('.humidity');
        const windEl = this.contentEl.querySelector('.wind');
        const visibilityEl = this.contentEl.querySelector('.visibility');

        if (feelsLikeEl) feelsLikeEl.textContent = `${weatherData.current.feels_like}${unitSymbol}`;
        if (humidityEl) humidityEl.textContent = `${weatherData.current.humidity}%`;
        if (windEl) {
            const windSpeed = this.config.units === 'imperial' ? 
                `${Math.round(weatherData.current.wind_speed * 0.621371)} mph` : 
                `${weatherData.current.wind_speed} km/h`;
            windEl.textContent = windSpeed;
        }
        if (visibilityEl) visibilityEl.textContent = `${weatherData.current.visibility} km`;

        // Update forecast
        if (weatherData.forecast && this.config.show_forecast) {
            this.displayForecast(weatherData.forecast);
        }
    }

    displayForecast(forecast) {
        const forecastList = this.contentEl.querySelector('.forecast-list');
        if (!forecastList) return;
        
        forecastList.innerHTML = '';
        
        forecast.forEach(day => {
            const forecastItem = document.createElement('div');
            forecastItem.className = 'forecast-item';
            
            const dayName = new Date(day.date).toLocaleDateString('en-US', { weekday: 'short' });
            const unitSymbol = this.getUnitSymbol();
            
            forecastItem.innerHTML = `
                <div class="forecast-day">${dayName}</div>
                <div class="forecast-emoji">${day.emoji}</div>
                <div class="forecast-temp">${day.max_temp}${unitSymbol}</div>
                <div class="forecast-temp-low">${day.min_temp}${unitSymbol}</div>
            `;
            
            forecastList.appendChild(forecastItem);
        });
    }

    getConditionFromCode(code) {
        // Map weather codes to general conditions for background styling
        const codeMap = {
            '113': 'clear',
            '116': 'clouds',
            '119': 'clouds',
            '122': 'clouds',
            '143': 'mist',
            '248': 'mist',
            '260': 'mist',
        };
        
        if (code.startsWith('2')) return 'thunderstorm';
        if (code.startsWith('3')) return 'rain';
        if (code.startsWith('5')) return 'rain';
        if (code.startsWith('6')) return 'snow';
        
        return codeMap[code] || 'clouds';
    }

    getUnitSymbol() {
        return this.config.units === 'imperial' ? '°F' : '°C';
    }

    updateLastUpdated() {
        if (this.lastUpdatedEl) {
            const now = new Date();
            this.lastUpdatedEl.textContent = now.toLocaleTimeString();
        }
    }

    showLoading() {
        this.loadingEl.style.display = 'block';
        this.errorEl.style.display = 'none';
        this.contentEl.style.display = 'none';
    }

    showError(message) {
        this.loadingEl.style.display = 'none';
        this.contentEl.style.display = 'none';
        this.errorEl.style.display = 'block';
        
        const errorMessage = this.errorEl.querySelector('.error-message');
        if (errorMessage) {
            errorMessage.textContent = message;
        }
    }

    showContent() {
        this.loadingEl.style.display = 'none';
        this.errorEl.style.display = 'none';
        this.contentEl.style.display = 'block';
    }

    // Public method to manually refresh the widget
    refresh() {
        this.loadWeatherData();
    }
}

// Initialize all weather widgets when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    const widgets = document.querySelectorAll('.weather-widget');
    
    widgets.forEach(widget => {
        new WeatherWidget(widget);
    });
});

// Re-initialize widgets after Elementor preview updates
if (typeof elementorFrontend !== 'undefined') {
    elementorFrontend.hooks.addAction('frontend/element_ready/weather_widget.default', function($scope) {
        const widget = $scope.find('.weather-widget')[0];
        if (widget) {
            new WeatherWidget(widget);
        }
    });
}

// Expose WeatherWidget globally for manual initialization
window.WeatherWidget = WeatherWidget;
