<?php
/**
 * Weather Widget for Elementor
 *
 * @package RelayoutDesigns
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Make sure Elementor is active
if (!class_exists('\Elementor\Widget_Base')) {
    return;
}

class Weather_Widget extends \Elementor\Widget_Base {

    public function get_name() {
        return 'weather_widget';
    }

    public function get_title() {
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            return __('Weather Widget', 'relayout-design') . ' <span style="color: #ff6b35; font-weight: bold; font-size: 10px; background: #fff3f0; padding: 2px 6px; border-radius: 3px; margin-left: 5px;">PRO</span>';
        }
        return __('Weather Widget', 'relayout-design');
    }

    public function get_icon() {
        return 'eicon-globe';
    }

    public function get_categories() {
        return ['relayout-designs'];
    }

    public function get_script_depends() {
        return ['weather-widget-script'];
    }

    protected function register_controls() {
        // Info Section
        $this->start_controls_section(
            'info_section',
            [
                'label' => __('Weather Information', 'relayout-design'),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'weather_info_notice',
            [
                'type' => \Elementor\Controls_Manager::RAW_HTML,
                'raw' => '<div style="background: #e8f5e8; padding: 15px; border-radius: 5px; margin-bottom: 15px;">
                    <h4 style="margin: 0 0 10px 0; color: #2e7d32;">✅ No API Key Required!</h4>
                    <p style="margin: 0; font-size: 13px; line-height: 1.4;">
                        This weather widget uses free, open-source weather data.<br>
                        Just enter your city name and you\'re ready to go!
                    </p>
                </div>',
            ]
        );

        $this->end_controls_section();

        // Location Settings Section
        $this->start_controls_section(
            'location_section',
            [
                'label' => __('Location Settings', 'relayout-design'),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'city_name',
            [
                'label' => __('City Name', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT2,
                'default' => 'New York',
                'options' => [
                    'New York' => 'New York, USA',
                    'London' => 'London, UK',
                    'Paris' => 'Paris, France',
                    'Tokyo' => 'Tokyo, Japan',
                    'Sydney' => 'Sydney, Australia',
                    'Dubai' => 'Dubai, UAE',
                    'Singapore' => 'Singapore',
                    'Mumbai' => 'Mumbai, India',
                    'Delhi' => 'Delhi, India',
                    'Bangalore' => 'Bangalore, India',
                    'Hyderabad' => 'Hyderabad, India',
                    'Chennai' => 'Chennai, India',
                    'Kolkata' => 'Kolkata, India',
                    'Pune' => 'Pune, India',
                    'Ahmedabad' => 'Ahmedabad, India',
                    'Jaipur' => 'Jaipur, India',
                    'Lucknow' => 'Lucknow, India',
                    'Nagpur' => 'Nagpur, India',
                    'Indore' => 'Indore, India',
                    'Bhopal' => 'Bhopal, India',
                    'Visakhapatnam' => 'Visakhapatnam, India',
                    'Patna' => 'Patna, India',
                    'Vadodara' => 'Vadodara, India',
                    'Ludhiana' => 'Ludhiana, India',
                    'Agra' => 'Agra, India',
                    'Varanasi' => 'Varanasi, India',
                    'Srinagar' => 'Srinagar, India',
                    'Amritsar' => 'Amritsar, India',
                    'Ranchi' => 'Ranchi, India',
                    'Coimbatore' => 'Coimbatore, India',
                    'Jodhpur' => 'Jodhpur, India',
                    'Madurai' => 'Madurai, India',
                    'Chandigarh' => 'Chandigarh, India',
                    'Guwahati' => 'Guwahati, India',
                    'Mysore' => 'Mysore, India',
                    'Bhubaneswar' => 'Bhubaneswar, India',
                    'Thiruvananthapuram' => 'Thiruvananthapuram, India',
                    'Noida' => 'Noida, India',
                    'Jamshedpur' => 'Jamshedpur, India',
                    'Kochi' => 'Kochi, India',
                    'Dehradun' => 'Dehradun, India',
                    'Ajmer' => 'Ajmer, India',
                    'Siliguri' => 'Siliguri, India',
                    'Jammu' => 'Jammu, India',
                    'Mangalore' => 'Mangalore, India',
                    'Udaipur' => 'Udaipur, India',
                    'Bangkok' => 'Bangkok, Thailand',
                    'Phuket' => 'Phuket, Thailand',
                    'Chiang Mai' => 'Chiang Mai, Thailand',
                    'Pattaya' => 'Pattaya, Thailand',
                    'Taipei' => 'Taipei, Taiwan',
                    'Kaohsiung' => 'Kaohsiung, Taiwan',
                    'Seoul' => 'Seoul, South Korea',
                    'Busan' => 'Busan, South Korea',
                    'Jeju' => 'Jeju, South Korea',
                    'Osaka' => 'Osaka, Japan',
                    'Kyoto' => 'Kyoto, Japan',
                    'Hiroshima' => 'Hiroshima, Japan',
                    'Sapporo' => 'Sapporo, Japan',
                    'Yokohama' => 'Yokohama, Japan',
                    'Beijing' => 'Beijing, China',
                    'Shanghai' => 'Shanghai, China',
                    'Guangzhou' => 'Guangzhou, China',
                    'Shenzhen' => 'Shenzhen, China',
                    'Chengdu' => 'Chengdu, China',
                    'Xian' => 'Xian, China',
                    'Hangzhou' => 'Hangzhou, China',
                    'Nanjing' => 'Nanjing, China',
                    'Wuhan' => 'Wuhan, China',
                    'Tianjin' => 'Tianjin, China',
                    'Qingdao' => 'Qingdao, China',
                    'Kunming' => 'Kunming, China',
                    'Xiamen' => 'Xiamen, China',
                    'Kuala Lumpur' => 'Kuala Lumpur, Malaysia',
                    'Penang' => 'Penang, Malaysia',
                    'Johor Bahru' => 'Johor Bahru, Malaysia',
                    'Kota Kinabalu' => 'Kota Kinabalu, Malaysia',
                    'Langkawi' => 'Langkawi, Malaysia',
                    'Jakarta' => 'Jakarta, Indonesia',
                    'Bali' => 'Bali, Indonesia',
                    'Yogyakarta' => 'Yogyakarta, Indonesia',
                    'Surabaya' => 'Surabaya, Indonesia',
                    'Bandung' => 'Bandung, Indonesia',
                    'Medan' => 'Medan, Indonesia',
                    'Manila' => 'Manila, Philippines',
                    'Cebu' => 'Cebu, Philippines',
                    'Davao' => 'Davao, Philippines',
                    'Quezon City' => 'Quezon City, Philippines',
                    'Makati' => 'Makati, Philippines',
                    'Yangon' => 'Yangon, Myanmar',
                    'Mandalay' => 'Mandalay, Myanmar',
                    'Phnom Penh' => 'Phnom Penh, Cambodia',
                    'Siem Reap' => 'Siem Reap, Cambodia',
                    'Vientiane' => 'Vientiane, Laos',
                    'Luang Prabang' => 'Luang Prabang, Laos',
                    'Ho Chi Minh City' => 'Ho Chi Minh City, Vietnam',
                    'Hanoi' => 'Hanoi, Vietnam',
                    'Sapa' => 'Sapa, Vietnam',
                    'Da Nang' => 'Da Nang, Vietnam',
                    'Hoi An' => 'Hoi An, Vietnam',
                    'Nha Trang' => 'Nha Trang, Vietnam',
                    'Hue' => 'Hue, Vietnam',
                    'Can Tho' => 'Can Tho, Vietnam',
                    'Dalat' => 'Dalat, Vietnam',
                    'Hai Phong' => 'Hai Phong, Vietnam',
                    'Hong Kong' => 'Hong Kong',
                    'Macau' => 'Macau',
                ],
                'select2options' => [
                    'tags' => true,
                    'placeholder' => __('Select or type city name...', 'relayout-design'),
                ],
                'description' => __('Select from popular cities or type any city name. Format: City or City,Country (e.g., London,UK). Note: Weather data will only display on published pages or preview mode, not in the Elementor editor.', 'relayout-design'),
            ]
        );

        $this->end_controls_section();

        // Display Settings Section
        $this->start_controls_section(
            'display_section',
            [
                'label' => __('Display Settings', 'relayout-design'),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'units',
            [
                'label' => __('Temperature Units', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'metric',
                'options' => [
                    'metric' => __('Celsius (°C)', 'relayout-design'),
                    'imperial' => __('Fahrenheit (°F)', 'relayout-design'),
                ],
            ]
        );

        $this->add_control(
            'show_forecast',
            [
                'label' => __('Show 3-Day Forecast', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'update_interval',
            [
                'label' => __('Update Interval (minutes)', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 30,
                'min' => 15,
                'max' => 120,
                'step' => 15,
            ]
        );

        $this->end_controls_section();

        // Style Section - Main Weather
        $this->start_controls_section(
            'main_weather_style',
            [
                'label' => __('Main Weather Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'background_color',
            [
                'label' => __('Background Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .weather-widget' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'temperature_color',
            [
                'label' => __('Temperature Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}} .current-temp' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'location_color',
            [
                'label' => __('Location Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#666666',
                'selectors' => [
                    '{{WRAPPER}} .location' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'temperature_typography',
                'label' => __('Temperature Typography', 'relayout-design'),
                'selector' => '{{WRAPPER}} .current-temp',
            ]
        );

        $this->end_controls_section();

        // Style Section - Forecast
        $this->start_controls_section(
            'forecast_style',
            [
                'label' => __('Forecast Style', 'relayout-design'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_forecast' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'forecast_background',
            [
                'label' => __('Forecast Background', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#f8f9fa',
                'selectors' => [
                    '{{WRAPPER}} .forecast-item' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'forecast_text_color',
            [
                'label' => __('Forecast Text Color', 'relayout-design'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}} .forecast-item' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        // Check if premium features are available
        if (!function_exists('relayout_designs_is_premium_available') || !relayout_designs_is_premium_available()) {
            $upgrade_prompt = relayout_designs_show_upgrade_prompt('Weather Widget');
            if ($upgrade_prompt) {
                echo $upgrade_prompt;
                return;
            }
        }
        
        $settings = $this->get_settings_for_display();
        
        // Prepare widget data
        $widget_data = [
            'city_name' => $settings['city_name'],
            'units' => $settings['units'],
            'show_forecast' => $settings['show_forecast'] === 'yes',
            'update_interval' => (int) $settings['update_interval'],
        ];

        $unit_symbol = $settings['units'] === 'imperial' ? '°F' : '°C';
        ?>
        <div class="weather-widget" data-config="<?php echo esc_attr(json_encode($widget_data)); ?>">
            <div class="weather-loading">
                <div class="loading-spinner"></div>
                <p><?php echo esc_html__('Loading weather data...', 'relayout-design'); ?></p>
            </div>
            
            <div class="weather-error" style="display: none;">
                <div class="error-icon">⚠️</div>
                <p class="error-message"><?php echo esc_html__('Unable to load weather data', 'relayout-design'); ?></p>
            </div>

            <div class="weather-content" style="display: none;">
                <div class="current-weather">
                    <div class="weather-icon">
                        <div class="weather-emoji">☀️</div>
                    </div>
                    <div class="weather-info">
                        <div class="current-temp">--<?php echo esc_html($unit_symbol); ?></div>
                        <div class="weather-description">--</div>
                        <div class="location">--</div>
                    </div>
                </div>

                <div class="weather-details">
                    <div class="detail-item">
                        <span class="detail-label"><?php echo esc_html__('Feels like', 'relayout-design'); ?></span>
                        <span class="detail-value feels-like">--<?php echo esc_html($unit_symbol); ?></span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label"><?php echo esc_html__('Humidity', 'relayout-design'); ?></span>
                        <span class="detail-value humidity">--%</span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label"><?php echo esc_html__('Wind', 'relayout-design'); ?></span>
                        <span class="detail-value wind">-- <?php echo $settings['units'] === 'imperial' ? 'mph' : 'km/h'; ?></span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label"><?php echo esc_html__('Visibility', 'relayout-design'); ?></span>
                        <span class="detail-value visibility">-- km</span>
                    </div>
                </div>

                <?php if ($settings['show_forecast'] === 'yes'): ?>
                <div class="weather-forecast">
                    <h4 class="forecast-title"><?php echo esc_html__('3-Day Forecast', 'relayout-design'); ?></h4>
                    <div class="forecast-list">
                        <!-- Forecast items will be populated by JavaScript -->
                    </div>
                </div>
                <?php endif; ?>
            </div>

            <div class="last-updated">
                <small><?php echo esc_html__('Last updated:', 'relayout-design'); ?> <span class="update-time">--</span></small>
            </div>
        </div>
        <?php
    }
}
