<?php
/**
 * Plugin Name: Relayout Designs
 * Plugin URI:  https://relayout.in/
 * Description: A dynamic history timeline addon for Elementor.
 * Version:     3.0.18
 * Author:      Relayout Designs
 * Author URI:  https://relayout.in/
 * Text Domain: relayout-design
 * 
 * @package RelayoutDesigns
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('RELAYOUT_DESIGNS_VERSION', '3.0.18');
define('RELAYOUT_DESIGNS_PLUGIN_FILE', __FILE__);
define('RELAYOUT_DESIGNS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('RELAYOUT_DESIGNS_PLUGIN_URL', plugin_dir_url(__FILE__));

// Define RLUS_VERSION constant for SDK compatibility
if (!defined('RLUS_VERSION')) {
    define('RLUS_VERSION', '1.0.0');
}

// Development mode - set to false for production
define('RELAYOUT_DESIGNS_DEV_MODE', false);

// License checking - with error handling
$sdk_file = plugin_dir_path(__FILE__) . 'sdk/class-rlus-sdk.php';
if (file_exists($sdk_file)) {
    require_once $sdk_file;
} else {
    // Fallback if SDK file is missing
    add_action('admin_notices', function() {
        echo '<div class="notice notice-error"><p>Relayout Designs: SDK file missing. Please reinstall the plugin.</p></div>';
    });
    return;
}

class Relayout_Designs_License_Check {
    private $sdk;
    
    public function __construct() {
        // Check if SDK class exists
        if (!class_exists('Rlus_Sdk')) {
            return;
        }
        
        // Use your RLUS server URL - replace with your actual domain
        $license_server_url = 'https://relayout.in/wp-admin/admin-ajax.php';
        $this->sdk = new Rlus_Sdk('plugin_relayout-designs', $license_server_url);
        add_action('admin_notices', array($this, 'license_notice'));
        add_action('admin_menu', array($this, 'license_page'));
        add_action('admin_init', array($this, 'init_updater'));
    }
    
    public function license_notice() {
        // Skip notices in development mode
        if (defined('RELAYOUT_DESIGNS_DEV_MODE') && RELAYOUT_DESIGNS_DEV_MODE) {
            echo '<div class="notice notice-info"><p>Relayout Designs: Development mode active - all features unlocked.</p></div>';
            return;
        }
        
        // Get license status information
        $stored_license = $this->sdk->get_stored_license_key();
        $is_valid = $this->sdk->is_license_valid();
        $is_expired = $this->sdk->is_license_expired();
        $needs_renewal = $this->sdk->needs_renewal();
        $days_remaining = $this->sdk->get_days_remaining();
        $status_text = $this->sdk->get_status_text();
        
        if (!$is_valid) {
            if (empty($stored_license)) {
                // No license key entered
                echo '<div class="notice notice-warning"><p><strong>Relayout Designs:</strong> Premium features require a license key. <a href="' . admin_url('options-general.php?page=relayout-license') . '">Enter License Key</a> | <a href="https://relayout.in/pricing" target="_blank">Get License</a></p></div>';
            } else {
                // License key exists but is invalid - check if expired
                if ($is_expired) {
                    $message = '<strong>Relayout Designs:</strong> Your license has expired.';
                    if ($status_text) {
                        $message .= ' (' . esc_html($status_text) . ')';
                    }
                    $message .= ' <a href="https://relayout.in/pricing" target="_blank">Renew License</a> to restore premium features.';
                    echo '<div class="notice notice-error"><p>' . $message . '</p></div>';
                } else {
                    // Invalid for other reasons (connection issues, invalid key, etc.)
                    $message = '<strong>Relayout Designs:</strong> License validation failed.';
                    if ($status_text) {
                        $message .= ' (' . esc_html($status_text) . ')';
                    }
                    $message .= ' <a href="' . admin_url('options-general.php?page=relayout-license') . '">Check License</a>';
                    echo '<div class="notice notice-error"><p>' . $message . '</p></div>';
                }
            }
        } else {
            // License is valid, check if renewal is needed
            if ($needs_renewal && $days_remaining !== false) {
                $message = '<strong>Relayout Designs:</strong> Your license expires in ' . $days_remaining . ' day' . ($days_remaining != 1 ? 's' : '') . '.';
                if ($status_text) {
                    $message .= ' (' . esc_html($status_text) . ')';
                }
                $message .= ' <a href="https://relayout.in/pricing" target="_blank">Renew Now</a> to avoid service interruption.';
                echo '<div class="notice notice-warning"><p>' . $message . '</p></div>';
            }
        }
    }
    
    public function license_page() {
        add_options_page(
            'Relayout Designs License', 
            'Relayout License', 
            'manage_options', 
            'relayout-license', 
            array($this, 'render_license_page_with_updates')
        );
    }
    
    public function render_license_page_with_updates() {
        // Render the original license page
        $this->sdk->render_license_page();
        
        // Add force update section
        ?>
        <div class="wrap" style="margin-top: 20px;">
            <div class="card">
                <h2 class="title">🔄 Force Update Check</h2>
                <p>If you don't see the latest plugin updates, click the button below to force WordPress to check for updates immediately.</p>
                <button type="button" id="relayout-force-update" class="button button-secondary">
                    Force Update Check
                </button>
                <div id="relayout-update-message" style="margin-top: 10px;"></div>
            </div>
        </div>
        
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            $('#relayout-force-update').click(function() {
                var button = $(this);
                var messageDiv = $('#relayout-update-message');
                
                button.prop('disabled', true).text('Checking for updates...');
                messageDiv.html('');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'relayout_force_update',
                        nonce: '<?php echo wp_create_nonce('relayout_force_update'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            messageDiv.html('<div class="notice notice-success"><p>' + response.data.message + '</p></div>');
                            // Redirect to plugins page after 2 seconds
                            setTimeout(function() {
                                window.location.href = '<?php echo admin_url('plugins.php'); ?>';
                            }, 2000);
                        } else {
                            messageDiv.html('<div class="notice notice-error"><p>Error: ' + response.data + '</p></div>');
                        }
                    },
                    error: function() {
                        messageDiv.html('<div class="notice notice-error"><p>Failed to force update check. Please try again.</p></div>');
                    },
                    complete: function() {
                        button.prop('disabled', false).text('Force Update Check');
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    public function is_licensed() {
        // Development mode bypass
        if (defined('RELAYOUT_DESIGNS_DEV_MODE') && RELAYOUT_DESIGNS_DEV_MODE) {
            return true;
        }
        return $this->sdk->is_license_valid();
    }
    
    /**
     * Get SDK instance for advanced license operations
     *
     * @return RLUS_SDK
     */
    public function get_sdk() {
        return $this->sdk;
    }
    
    public function init_updater() {
        new Relayout_Designs_Updater(RELAYOUT_DESIGNS_PLUGIN_FILE, RELAYOUT_DESIGNS_VERSION, $this->sdk);
    }
}

// Auto-updater class
class Relayout_Designs_Updater {
    private $plugin_file;
    private $version;
    private $sdk;
    private $plugin_slug;
    
    public function __construct($plugin_file, $version, $sdk) {
        $this->plugin_file = $plugin_file;
        $this->version = $version;
        $this->sdk = $sdk;
        $this->plugin_slug = plugin_basename($plugin_file);
        
        add_filter('pre_set_site_transient_update_plugins', array($this, 'check_updates'));
        add_filter('plugins_api', array($this, 'plugin_info'), 20, 3);
    }
    
    /**
     * Get the correct plugin directory name for updates
     * Always returns 'relayout-designs' regardless of current folder name
     */
    private function get_plugin_directory_name() {
        return 'relayout-designs';
    }
    
    public function check_updates($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }
        
        $update = $this->sdk->check_for_updates($this->version);
        
        if ($update && version_compare($this->version, $update['version'], '<')) {
            $transient->response[$this->plugin_slug] = (object) array(
                'slug' => $this->get_plugin_directory_name(),
                'plugin' => $this->get_plugin_directory_name() . '/relayout-designs.php',
                'new_version' => $update['version'],
                'package' => $update['download_url'],
                'url' => $update['details_url'] ?? '',
                'tested' => $update['tested'] ?? '',
                'requires_php' => $update['requires_php'] ?? '',
                'compatibility' => new stdClass()
            );
        }
        
        return $transient;
    }
    
    public function plugin_info($result, $action, $args) {
        if ($action !== 'plugin_information') {
            return $result;
        }
        
        // Check for both current slug and correct slug
        $current_slug = dirname($this->plugin_slug);
        $correct_slug = $this->get_plugin_directory_name();
        
        if ($args->slug !== $current_slug && $args->slug !== $correct_slug) {
            return $result;
        }
        
        $update = $this->sdk->check_for_updates($this->version);
        
        if ($update) {
            return (object) array(
                'name' => 'Relayout Designs',
                'slug' => $this->get_plugin_directory_name(),
                'version' => $update['version'],
                'author' => 'Relayout Designs',
                'homepage' => 'https://relayout.in/',
                'short_description' => 'A dynamic history timeline addon for Elementor.',
                'sections' => array(
                    'description' => $update['description'] ?? 'A dynamic history timeline addon for Elementor.',
                    'changelog' => $update['changelog'] ?? 'Bug fixes and improvements.'
                ),
                'download_link' => $update['download_url'],
                'tested' => $update['tested'] ?? '',
                'requires_php' => $update['requires_php'] ?? '',
                'last_updated' => $update['last_updated'] ?? ''
            );
        }
        
        return $result;
    }
}

// Initialize license check after WordPress is loaded
function relayout_designs_init_license() {
    global $relayout_designs_license;
    $relayout_designs_license = new Relayout_Designs_License_Check();
}
add_action('init', 'relayout_designs_init_license');

// Auto-force update check on plugin activation
register_activation_hook(__FILE__, 'relayout_designs_activation_force_update');
function relayout_designs_activation_force_update() {
    // Force update check when plugin is activated
    relayout_designs_force_update_check();
}

// Check for version changes and force update if needed
add_action('admin_init', 'relayout_designs_check_version_change');
function relayout_designs_check_version_change() {
    $stored_version = get_option('relayout_designs_version', '0.0.0');
    $current_version = RELAYOUT_DESIGNS_VERSION;
    
    if (version_compare($stored_version, $current_version, '<')) {
        // Version has changed, force update check
        relayout_designs_force_update_check();
        update_option('relayout_designs_version', $current_version);
        
        // Show update success notice
        add_action('admin_notices', function() {
            echo '<div class="notice notice-success is-dismissible">
                <p><strong>Relayout Designs:</strong> Plugin updated successfully! Update check has been refreshed.</p>
            </div>';
        });
    }
}

// Add admin notice for available updates
add_action('admin_notices', 'relayout_designs_update_available_notice');
function relayout_designs_update_available_notice() {
    global $relayout_designs_license;
    
    if (!current_user_can('update_plugins') || !isset($relayout_designs_license)) {
        return;
    }
    
    // Check if there's an update available
    $update = $relayout_designs_license->get_sdk()->check_for_updates(RELAYOUT_DESIGNS_VERSION);
    
    if ($update && version_compare(RELAYOUT_DESIGNS_VERSION, $update['version'], '<')) {
        $plugins_url = admin_url('plugins.php');
        $license_url = admin_url('options-general.php?page=relayout-license');
        
        echo '<div class="notice notice-info">
            <p><strong>🚀 Relayout Designs Update Available!</strong></p>
            <p>Version <strong>' . esc_html($update['version']) . '</strong> is now available. 
            <a href="' . esc_url($plugins_url) . '">Update now</a> or 
            <a href="' . esc_url($license_url) . '">Check for Updates now</a> if you don\'t see it.</p>
        </div>';
    }
}

// Helper function to check if premium features are available
function relayout_designs_is_premium_available() {
    global $relayout_designs_license;
    if (!isset($relayout_designs_license)) {
        return false;
    }
    return $relayout_designs_license->is_licensed();
}

// Helper function for premium feature protection
function relayout_designs_premium_feature($fallback_message = 'License required for this feature.') {
    if (!relayout_designs_is_premium_available()) {
        echo '<div class="relayout-premium-notice">' . esc_html($fallback_message) . '</div>';
        return false;
    }
    return true;
}

// Helper function to add Pro badge to widget titles
function relayout_designs_add_pro_badge($widget_title) {
    if (!relayout_designs_is_premium_available()) {
        return $widget_title . ' <span class="relayout-pro-badge">PRO</span>';
    }
    return $widget_title;
}

// Force update check function
function relayout_designs_force_update_check() {
    // Clear plugin update transients to force fresh check
    delete_site_transient('update_plugins');
    delete_transient('update_plugins');
    
    // Clear any plugin-specific update cache
    delete_option('relayout_designs_update_cache');
    delete_transient('relayout_designs_update_check');
    
    // Force WordPress to check for updates immediately
    wp_update_plugins();
    
    return true;
}

// Add admin action to force update check
add_action('wp_ajax_relayout_force_update', 'relayout_designs_ajax_force_update');
function relayout_designs_ajax_force_update() {
    // Verify user permissions
    if (!current_user_can('update_plugins')) {
        wp_die('Insufficient permissions');
    }
    
    // Verify nonce for security
    check_ajax_referer('relayout_force_update', 'nonce');
    
    // Force update check
    relayout_designs_force_update_check();
    
    wp_send_json_success(array(
        'message' => 'Update check forced successfully. Please refresh the plugins page.'
    ));
}

// Helper function to show upgrade prompt in widget content
function relayout_designs_show_upgrade_prompt($widget_name) {
    if (!relayout_designs_is_premium_available()) {
        return '
        <div class="relayout-upgrade-prompt" style="
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 40px 30px;
            text-align: center;
            border-radius: 12px;
            margin: 20px 0;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
            position: relative;
            overflow: hidden;
        ">
            <div class="relayout-upgrade-content" style="position: relative; z-index: 2;">
                <div style="font-size: 48px; margin-bottom: 15px;">🚀</div>
                <h3 style="margin: 0 0 15px 0; font-size: 24px; font-weight: 600;">Unlock ' . esc_html($widget_name) . '</h3>
                <p style="margin: 0 0 25px 0; opacity: 0.9; font-size: 16px;">This is a premium widget. Get your license to unlock all powerful features.</p>
                <a href="' . admin_url('options-general.php?page=relayout-license') . '" class="relayout-upgrade-btn" style="
                    display: inline-block;
                    background: #ff6b35;
                    color: white;
                    padding: 12px 30px;
                    text-decoration: none;
                    border-radius: 25px;
                    font-weight: 600;
                    transition: all 0.3s ease;
                    box-shadow: 0 4px 15px rgba(255, 107, 53, 0.4);
                " onmouseover="this.style.transform=\'scale(1.05)\'; this.style.boxShadow=\'0 6px 20px rgba(255, 107, 53, 0.6)\'" onmouseout="this.style.transform=\'scale(1)\'; this.style.boxShadow=\'0 4px 15px rgba(255, 107, 53, 0.4)\'">
                    Get License Key
                </a>
            </div>
            <div style="
                position: absolute;
                top: -50%;
                right: -50%;
                width: 100%;
                height: 100%;
                background: rgba(255,255,255,0.1);
                border-radius: 50%;
                z-index: 1;
            "></div>
        </div>';
    }
    return false;
}

// Check if Elementor is active
function relayout_designs_check_elementor() {
    if (!did_action('elementor/loaded')) {
        add_action('admin_notices', function() {
            if (!current_user_can('activate_plugins')) {
                return;
            }
            ?>
            <div class="notice notice-error">
                <p><?php 
                    printf(
                        esc_html__('Relayout Designs requires %s to be installed and activated.', 'relayout-design'),
                        '<a href="' . esc_url('https://wordpress.org/plugins/elementor/') . '">Elementor</a>'
                    );
                ?></p>
            </div>
            <?php
        });
        return false;
    }
    return true;
}

// Initialize plugin only if Elementor is active
function relayout_designs_init() {
    if (!relayout_designs_check_elementor()) {
        return;
    }
    
    // Register Custom Elementor Category
    add_action('elementor/elements/categories_registered', 'add_relayout_designs_category');
    
    // Register widgets
    add_action('elementor/widgets/register', 'register_relayout_design_widgets');
    
    // Enqueue assets
    add_action('wp_enqueue_scripts', 'relayout_design_enqueue_assets');
}
add_action('plugins_loaded', 'relayout_designs_init');

// Register Custom Elementor Category
function add_relayout_designs_category($elements_manager) {
    $elements_manager->add_category(
        'relayout-designs',
        [
            'title' => __('Relayout Designs', 'relayout-design'),
            'icon'  => 'fa fa-plug',
        ]
    );
}

/**
 * Register Elementor Widgets
 * 
 * @param \Elementor\Widgets_Manager $widgets_manager Elementor widgets manager.
 * @return void
 */
function register_relayout_design_widgets($widgets_manager) {
    global $relayout_designs_license;
    
    // Check if license is valid for premium features
    $is_licensed = $relayout_designs_license->is_licensed();
    
    // Define which widgets are premium (require license)
    $premium_widgets = array(
        'checklist-manager-widget',
        'business-hours-widget',
        'weather-widget',
        'instagram-feed-widget',
        'age-calculator-widget',
        'cookie-consent-widget',
        'business-chat-widget',
        'countdown-timer-widget',
        'delivery-date-time-widget'
    );
    // Register History Timeline Widget
    $timeline_widget = plugin_dir_path(__FILE__) . 'includes/timeline-widget/timeline-widget.php';
    if (file_exists($timeline_widget)) {
        require_once $timeline_widget;
        if (class_exists('History_Timeline_Widget')) {
            $widgets_manager->register(new \History_Timeline_Widget());
        }
    }

    // Register Delivery Date and Time Widget (Premium)
    $delivery_widget = plugin_dir_path(__FILE__) . 'includes/delivery-date-time-widget/delivery-date-time-widget.php';
    if (file_exists($delivery_widget)) {
        require_once $delivery_widget;
        if (class_exists('Delivery_Date_Time_Widget')) {
            // Always register widget but show upgrade prompt if unlicensed
            $widgets_manager->register(new \Delivery_Date_Time_Widget());
        }
    }
    
    // Register Current Year Widget
    $current_year_widget = plugin_dir_path(__FILE__) . 'includes/current-year-widget/current-year-widget.php';
    if (file_exists($current_year_widget)) {
        require_once $current_year_widget;
        if (class_exists('Current_Year_Widget')) {
            $widgets_manager->register(new \Current_Year_Widget());
        }
    }

    // Register Checklist Manager Widget (Premium)
    $checklist_widget = plugin_dir_path(__FILE__) . 'includes/checklist-manager-widget/checklist-manager-widget.php';
    if (file_exists($checklist_widget)) {
        require_once $checklist_widget;
        if (class_exists('Checklist_Manager_Widget')) {
            // Always register widget but show upgrade prompt if unlicensed
            $widgets_manager->register(new \Checklist_Manager_Widget());
        }
    }

    // Register Business Hours Widget (Premium)
    $business_hours_widget = plugin_dir_path(__FILE__) . 'includes/business-hours-widget/business-hours-widget.php';
    if (file_exists($business_hours_widget)) {
        require_once $business_hours_widget;
        if (class_exists('Business_Hours_Widget')) {
            // Always register widget but show upgrade prompt if unlicensed
            $widgets_manager->register(new \Business_Hours_Widget());
        }
    }

    // Register Weather Widget (Premium)
    $weather_widget = plugin_dir_path(__FILE__) . 'includes/weather-widget/weather-widget.php';
    if (file_exists($weather_widget)) {
        require_once $weather_widget;
        if (class_exists('Weather_Widget')) {
            // Always register widget but show upgrade prompt if unlicensed
            $widgets_manager->register(new \Weather_Widget());
        }
    }

    // Register Disable Right Click Widget (Free)
    $disable_right_click_widget = plugin_dir_path(__FILE__) . 'includes/disable-right-click-widget/disable-right-click-widget.php';
    if (file_exists($disable_right_click_widget)) {
        require_once $disable_right_click_widget;
        if (class_exists('Disable_Right_Click_Widget')) {
            // Free widget - always register
            $widgets_manager->register(new \Disable_Right_Click_Widget());
        }
    }

    // Register Instagram Feed Widget (Premium)
    $instagram_feed_widget = plugin_dir_path(__FILE__) . 'includes/instagram-feed-widget/instagram-feed-widget.php';
    if (file_exists($instagram_feed_widget)) {
        require_once $instagram_feed_widget;
        if (class_exists('Instagram_Feed_Widget')) {
            // Always register widget but show upgrade prompt if unlicensed
            $widgets_manager->register(new \Instagram_Feed_Widget());
        }
    }

    // Register Age Calculator Widget (Premium)
    $age_calculator_widget = plugin_dir_path(__FILE__) . 'includes/age-calculator-widget/age-calculator-widget.php';
    if (file_exists($age_calculator_widget)) {
        require_once $age_calculator_widget;
        if (class_exists('Age_Calculator_Widget')) {
            // Always register widget but show upgrade prompt if unlicensed
            $widgets_manager->register(new \Age_Calculator_Widget());
        }
    }

    // Register Copy to Clipboard Widget (Free)
    $copy_to_clipboard_widget = plugin_dir_path(__FILE__) . 'includes/copy-to-clipboard-widget/copy-to-clipboard-widget.php';
    if (file_exists($copy_to_clipboard_widget)) {
        require_once $copy_to_clipboard_widget;
        if (class_exists('Copy_To_Clipboard_Widget')) {
            // Free widget - always register
            $widgets_manager->register(new \Copy_To_Clipboard_Widget());
        }
    }

    // Register Cookie Consent Widget (Premium)
    $cookie_consent_widget = plugin_dir_path(__FILE__) . 'includes/cookie-consent-widget/cookie-consent-widget.php';
    if (file_exists($cookie_consent_widget)) {
        require_once $cookie_consent_widget;
        if (class_exists('Cookie_Consent_Widget')) {
            // Always register widget but show upgrade prompt if unlicensed
            $widgets_manager->register(new \Cookie_Consent_Widget());
        }
    }

    // Register Business Chat Widget (Premium)
    $business_chat_widget = plugin_dir_path(__FILE__) . 'includes/business-chat-widget/business-chat-widget.php';
    if (file_exists($business_chat_widget)) {
        require_once $business_chat_widget;
        if (class_exists('Business_Chat_Widget')) {
            // Always register widget but show upgrade prompt if unlicensed
            $widgets_manager->register(new \Business_Chat_Widget());
        }
    }

    // Register FAQ Accordion Widget (Free)
    $faq_accordion_widget = plugin_dir_path(__FILE__) . 'includes/faq-accordion-widget/faq-accordion-widget.php';
    if (file_exists($faq_accordion_widget)) {
        require_once $faq_accordion_widget;
        if (class_exists('FAQ_Accordion_Widget')) {
            // Free widget - always register
            $widgets_manager->register(new \FAQ_Accordion_Widget());
        }
    }

    // Register Reading Progress Bar Widget (Free)
    $reading_progress_widget = plugin_dir_path(__FILE__) . 'includes/reading-progress-bar-widget/reading-progress-bar-widget.php';
    if (file_exists($reading_progress_widget)) {
        require_once $reading_progress_widget;
        if (class_exists('Reading_Progress_Bar_Widget')) {
            // Free widget - always register
            $widgets_manager->register(new \Reading_Progress_Bar_Widget());
        }
    }

    // Register Countdown Timer Widget (Premium)
    $countdown_timer_widget = plugin_dir_path(__FILE__) . 'includes/countdown-timer-widget/countdown-timer-widget.php';
    if (file_exists($countdown_timer_widget)) {
        require_once $countdown_timer_widget;
        if (class_exists('Countdown_Timer_Widget')) {
            // Always register widget but show upgrade prompt if unlicensed
            $widgets_manager->register(new \Countdown_Timer_Widget());
        }
    }
}

// Helper function to safely enqueue widget assets
function relayout_designs_enqueue_widget_assets($widget_name, $has_css = true, $has_js = true, $js_deps = []) {
    $widget_path = plugin_dir_path(__FILE__) . 'includes/' . $widget_name . '/assets/';
    $widget_url = plugin_dir_url(__FILE__) . 'includes/' . $widget_name . '/assets/';
    
    // Define possible asset names (handle inconsistent naming)
    $possible_names = [
        $widget_name, // Full widget name (e.g., 'checklist-manager-widget')
        str_replace('-widget', '', $widget_name), // Without '-widget' suffix (e.g., 'checklist-manager')
    ];
    
    // Enqueue CSS if exists
    if ($has_css) {
        $css_enqueued = false;
        foreach ($possible_names as $name) {
            $css_file = $widget_path . $name . '.css';
            if (file_exists($css_file)) {
                wp_enqueue_style(
                    $widget_name . '-style',
                    $widget_url . $name . '.css',
                    [],
                    RELAYOUT_DESIGNS_VERSION
                );
                $css_enqueued = true;
                break;
            }
        }
    }
    
    // Enqueue JS if exists
    if ($has_js) {
        $js_enqueued = false;
        foreach ($possible_names as $name) {
            $js_file = $widget_path . $name . '.js';
            if (file_exists($js_file)) {
                wp_enqueue_script(
                    $widget_name . '-script',
                    $widget_url . $name . '.js',
                    $js_deps,
                    RELAYOUT_DESIGNS_VERSION,
                    true
                );
                $js_enqueued = true;
                break;
            }
        }
    }
}

// Enqueue Styles and Scripts
function relayout_design_enqueue_assets() {
    // Timeline widget (special case - different CSS name)
    $timeline_css = plugin_dir_path(__FILE__) . 'includes/timeline-widget/assets/style.css';
    if (file_exists($timeline_css)) {
        wp_enqueue_style('history-timeline-style', plugin_dir_url(__FILE__) . 'includes/timeline-widget/assets/style.css', [], RELAYOUT_DESIGNS_VERSION);
    }

    // Delivery Date and Time widget (special case - different JS name)
    $delivery_js = plugin_dir_path(__FILE__) . 'includes/delivery-date-time-widget/assets/delivery-date-time.js';
    if (file_exists($delivery_js)) {
        wp_enqueue_script(
            'delivery-date-time-script',
            plugin_dir_url(__FILE__) . 'includes/delivery-date-time-widget/assets/delivery-date-time.js',
            [],
            RELAYOUT_DESIGNS_VERSION,
            true
        );
    }

    // Enqueue assets for all other widgets
    relayout_designs_enqueue_widget_assets('checklist-manager-widget');
    relayout_designs_enqueue_widget_assets('business-hours-widget');
    relayout_designs_enqueue_widget_assets('weather-widget');
    relayout_designs_enqueue_widget_assets('disable-right-click-widget');
    relayout_designs_enqueue_widget_assets('instagram-feed-widget');
    relayout_designs_enqueue_widget_assets('age-calculator-widget');
    relayout_designs_enqueue_widget_assets('copy-to-clipboard-widget');
    // Cookie Consent widget (special case - different asset names)
    $cookie_css = plugin_dir_path(__FILE__) . 'includes/cookie-consent-widget/assets/cookie-consent.css';
    if (file_exists($cookie_css)) {
        wp_enqueue_style('cookie-consent-style', plugin_dir_url(__FILE__) . 'includes/cookie-consent-widget/assets/cookie-consent.css', [], RELAYOUT_DESIGNS_VERSION);
    }
    
    $cookie_js = plugin_dir_path(__FILE__) . 'includes/cookie-consent-widget/assets/cookie-consent.js';
    if (file_exists($cookie_js)) {
        wp_enqueue_script('cookie-consent-script', plugin_dir_url(__FILE__) . 'includes/cookie-consent-widget/assets/cookie-consent.js', ['jquery'], RELAYOUT_DESIGNS_VERSION, true);
    }
    relayout_designs_enqueue_widget_assets('business-chat-widget', true, true, ['jquery']);
    relayout_designs_enqueue_widget_assets('faq-accordion-widget', true, true, ['jquery']);
    relayout_designs_enqueue_widget_assets('reading-progress-bar-widget');
    relayout_designs_enqueue_widget_assets('countdown-timer-widget');

}