<?php
/**
 * Relayout License & Updates System (RLUS) SDK
 * 
 * Production-ready SDK for integrating license validation and automatic updates
 * into WordPress plugins and themes.
 *
 * @package RLUS
 * @version 1.0.0
 * @author Relayout
 * @license GPL v2 or later
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Rlus_Sdk Class
 */
class Rlus_Sdk {

    /**
     * Product ID
     */
    private $product_id;

    /**
     * License server URL
     */
    private $license_server_url;

    /**
     * Site URL
     */
    private $site_url;

    /**
     * Constructor
     *
     * @param string $product_id
     * @param string $license_server_url
     */
    public function __construct($product_id, $license_server_url = '') {
        $this->product_id = $product_id;
        $this->license_server_url = $license_server_url ?: get_option('rlus_license_server_url', 'https://www.relayout.in');
        $this->site_url = home_url();
    }

    /**
     * Activate license
     *
     * @param string $license_key
     * @return array
     */
    public function activate_license($license_key) {
        $response = $this->make_request('activate', array(
            'license_key' => $license_key,
            'product_id' => $this->product_id,
            'site_url' => $this->site_url,
            'site_name' => get_bloginfo('name')
        ));

        if ($response['success']) {
            update_option($this->get_license_option_name(), $license_key);
            update_option($this->get_activation_option_name(), $response['activation_token'] ?? '');
            
            // Clear cache after successful activation
            $this->clear_license_cache($license_key);
            
            do_action('rlus_sdk_license_activated', $this->product_id, $license_key);
        } else {
            $check_response = $this->check_license($license_key);
            if (isset($check_response['is_expired']) && $check_response['is_expired'] === true) {
                $response['error'] = __('License has expired', 'rlus');
                if (isset($check_response['status_text']) && !empty($check_response['status_text'])) {
                    $response['error'] .= ': ' . $check_response['status_text'];
                }
            } elseif (isset($check_response['valid']) && $check_response['valid'] === false) {
                $response['error'] = $response['error'] ?? __('Invalid license key', 'rlus');
                if (isset($check_response['status_text']) && !empty($check_response['status_text'])) {
                    $response['error'] .= ': ' . $check_response['status_text'];
                }
            }
        }

        return $response;
    }

    /**
     * Deactivate license
     *
     * @param string $license_key
     * @return array
     */
    public function deactivate_license($license_key = '') {
        if (empty($license_key)) {
            $license_key = $this->get_stored_license_key();
        }

        $response = $this->make_request('deactivate', array(
            'license_key' => $license_key,
            'product_id' => $this->product_id,
            'site_url' => $this->site_url
        ));

        if ($response['success']) {
            // Clear cache before removing license data
            $this->clear_license_cache($license_key);
            
            // Remove stored license data
            delete_option($this->get_license_option_name());
            delete_option($this->get_activation_option_name());
            
            do_action('rlus_sdk_license_deactivated', $this->product_id, $license_key);
        }

        return $response;
    }

    /**
     * Check license status
     *
     * @param string $license_key
     * @return array
     */
    public function check_license($license_key = '') {
        if (empty($license_key)) {
            $license_key = $this->get_stored_license_key();
        }

        if (empty($license_key)) {
            return array(
                'valid' => false,
                'error' => __('No license key found', 'rlus')
            );
        }

        // Check cache first for better performance
        $cache_key = 'rlus_license_' . md5($this->product_id . $license_key . $this->site_url);
        $cached_result = get_transient($cache_key);
        
        if ($cached_result !== false && is_array($cached_result)) {
            return $cached_result;
        }

        $response = $this->make_request('check', array(
            'license_key' => $license_key,
            'product_id' => $this->product_id,
            'site_url' => $this->site_url
        ));

        // Smart caching: valid licenses cached longer, invalid shorter
        if (isset($response['valid']) && $response['valid']) {
            $cache_time = 6 * HOUR_IN_SECONDS; // 6 hours for valid licenses
        } else {
            $cache_time = 15 * MINUTE_IN_SECONDS; // 15 minutes for invalid
        }
        
        set_transient($cache_key, $response, $cache_time);
        do_action('rlus_sdk_license_checked', $this->product_id, $license_key, $response);

        return $response;
    }

    /**
     * Check if license is valid
     *
     * @return bool
     */
    public function is_license_valid() {
        $check = $this->check_license();
        return $check['valid'] ?? false;
    }

    /**
     * Check if license is expired
     *
     * @return bool
     */
    public function is_license_expired() {
        $check = $this->check_license();
        // Use new is_expired field from enhanced API response
        if (isset($check['is_expired'])) {
            return $check['is_expired'] === true;
        }
        // Fallback to legacy expired field for backward compatibility
        return isset($check['expired']) && $check['expired'] === true;
    }

    /**
     * Check if license needs renewal (expires soon)
     *
     * @param int $days_threshold Days before expiration to consider "needs renewal"
     * @return bool
     */
    public function needs_renewal($days_threshold = 30) {
        $days_remaining = $this->get_days_remaining();
        return $days_remaining !== false && $days_remaining <= $days_threshold && $days_remaining > 0;
    }

    /**
     * Get days remaining until license expires
     *
     * @return int|false Number of days remaining, or false if no expiration data
     */
    public function get_days_remaining() {
        $check = $this->check_license();
        
        // Use new days_remaining field from enhanced API response
        if (isset($check['days_remaining']) && is_numeric($check['days_remaining'])) {
            return max(0, intval($check['days_remaining']));
        }
        
        // Fallback to calculating from expires_at for backward compatibility
        if (!isset($check['expires_at']) || empty($check['expires_at'])) {
            return false;
        }
        
        $expires_timestamp = strtotime($check['expires_at']);
        if ($expires_timestamp === false) {
            return false;
        }
        
        $current_timestamp = current_time('timestamp');
        $days_remaining = ceil(($expires_timestamp - $current_timestamp) / DAY_IN_SECONDS);
        
        return max(0, $days_remaining);
    }

    /**
     * Get license expiration date in readable format
     *
     * @param string $format Date format (default: 'M j, Y')
     * @return string|false Formatted expiration date or false if no expiration
     */
    public function get_expiration_date($format = 'M j, Y') {
        $check = $this->check_license();
        
        if (!isset($check['expires_at']) || empty($check['expires_at'])) {
            return false;
        }
        
        $expires_timestamp = strtotime($check['expires_at']);
        if ($expires_timestamp === false) {
            return false;
        }
        
        return date($format, $expires_timestamp);
    }

    /**
     * Get license status text from API response
     *
     * @return string|false Status text or false if not available
     */
    public function get_status_text() {
        $check = $this->check_license();
        
        if (isset($check['status_text']) && !empty($check['status_text'])) {
            return $check['status_text'];
        }
        
        return false;
    }

    /**
     * Get enhanced license status information
     *
     * @return array Enhanced status information
     */
    public function get_license_status() {
        $check = $this->check_license();
        
        $status = array(
            'valid' => $check['valid'] ?? false,
            'is_expired' => $this->is_license_expired(),
            'days_remaining' => $this->get_days_remaining(),
            'status_text' => $this->get_status_text(),
            'expires_at' => $check['expires_at'] ?? null,
            'needs_renewal' => $this->needs_renewal()
        );
        
        return $status;
    }

    /**
     * Get license info
     *
     * @return array
     */
    public function get_license_info() {
        $license_key = $this->get_stored_license_key();
        
        if (empty($license_key)) {
            return array();
        }

        $check = $this->check_license($license_key);
        
        if (!$check['valid']) {
            return array();
        }

        return array(
            'license_key' => $license_key,
            'license_type' => $check['license_type'] ?? '',
            'expires_at' => $check['expires_at'] ?? '',
            'product_id' => $this->product_id
        );
    }

    /**
     * Get stored license key
     *
     * @return string
     */
    public function get_stored_license_key() {
        return get_option($this->get_license_option_name(), '');
    }

    /**
     * Clear license cache
     *
     * @param string $license_key Optional specific license key to clear
     */
    public function clear_license_cache($license_key = '') {
        if (empty($license_key)) {
            $license_key = $this->get_stored_license_key();
        }
        
        if (!empty($license_key)) {
            $cache_key = 'rlus_license_' . md5($this->product_id . $license_key . $this->site_url);
            delete_transient($cache_key);
        }
    }

    /**
     * Check for updates
     *
     * @param string $current_version
     * @return array|false
     */
    public function check_for_updates($current_version) {
        $license_key = $this->get_stored_license_key();
        
        if (empty($license_key)) {
            return false;
        }

        $response = $this->make_request('check_updates', array(
            'license_key' => $license_key,
            'product_id' => $this->product_id,
            'site_url' => $this->site_url,
            'version' => $current_version
        ));

        // Handle different response formats safely
        if (!is_array($response)) {
            return false;
        }

        // Check if response indicates failure
        if (isset($response['success']) && $response['success'] === false) {
            return false;
        }

        // For direct API responses (no success wrapper)
        if (isset($response['version'])) {
            $data = $response;
        } else {
            // For wrapped responses
            $data = $response['data'] ?? array();
        }
        
        if (empty($data['version'])) {
            return false;
        }

        if (version_compare($current_version, $data['version'], '>=')) {
            return false;
        }

        return $data;
    }

    /**
     * Make API request
     *
     * @param string $action
     * @param array $data
     * @return array
     */
    private function make_request($action, $data = array()) {
        if (empty($this->license_server_url)) {
            return array(
                'success' => false,
                'error' => __('License server URL not configured', 'rlus')
            );
        }

        // Construct proper AJAX URL for RLUS
        $api_url = rtrim($this->license_server_url, '/');
        
        // Map actions to RLUS AJAX actions
        $ajax_action_map = array(
            'activate' => 'rlus_activate_license',
            'deactivate' => 'rlus_deactivate_license', 
            'check' => 'rlus_check_license',
            'check_updates' => 'rlus_check_update'
        );
        
        $ajax_action = isset($ajax_action_map[$action]) ? $ajax_action_map[$action] : 'rlus_' . str_replace('/', '_', $action);
        $data['action'] = $ajax_action;
        
        // Only create nonce if WordPress is fully loaded
        if (function_exists('wp_create_nonce')) {
            $data['nonce'] = wp_create_nonce('rlus_ajax');
        }
        
        $response = wp_remote_post($api_url, array(
            'timeout' => 30,
            'body' => $data,
            'headers' => array(
                'User-Agent' => 'RLUS-SDK/' . (defined('RLUS_VERSION') ? RLUS_VERSION : '1.0.0') . '; ' . $this->site_url,
                'Accept' => 'application/json',
                'Content-Type' => 'application/x-www-form-urlencoded'
            ),
            'sslverify' => true,
            'redirection' => 5
        ));

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error' => __('Connection failed. Please check your internet connection and try again.', 'rlus')
            );
        }

        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code !== 200) {
            return array(
                'success' => false,
                'error' => sprintf(__('Server error (HTTP %d)', 'rlus'), $response_code)
            );
        }

        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE || !is_array($result)) {
            return array(
                'success' => false,
                'error' => __('Invalid server response. Please try again.', 'rlus')
            );
        }

        // Handle different response formats
        if (isset($result['success']) && $result['success'] === false) {
            if (!isset($result['error']) || empty(trim($result['error']))) {
                $result['error'] = __('License server returned an error', 'rlus');
            }
            return $result;
        }
        
        // For RLUS API responses that return data directly (like license checks)
        if (isset($result['valid']) && !isset($result['success'])) {
            // This is a direct license check response - return as-is for compatibility
            return $result;
        }
        
        // For update check responses
        if (isset($result['version']) && !isset($result['success'])) {
            return array(
                'success' => !isset($result['error']),
                'data' => $result,
                'error' => $result['error'] ?? null
            );
        }

        return $result;
    }

    /**
     * Get license option name
     *
     * @return string
     */
    private function get_license_option_name() {
        return 'rlus_license_' . md5($this->product_id);
    }

    /**
     * Get activation option name
     *
     * @return string
     */
    private function get_activation_option_name() {
        return 'rlus_activation_' . md5($this->product_id);
    }

    /**
     * Add license check to admin notices
     */
    public function add_license_notice() {
        if (!$this->is_license_valid()) {
            add_action('admin_notices', array($this, 'show_license_notice'));
        }
    }

    /**
     * Show license notice
     */
    public function show_license_notice() {
        $product_name = apply_filters('rlus_sdk_product_name', $this->product_id, $this->product_id);
        $stored_license = $this->get_stored_license_key();
        $is_expired = $this->is_license_expired();
        $status_text = $this->get_status_text();
        
        if (!empty($stored_license) && $is_expired) {
            // License exists but is expired
            $message = sprintf(
                __('%s license has expired.', 'rlus'),
                esc_html($product_name)
            );
            if ($status_text) {
                $message .= ' (' . esc_html($status_text) . ')';
            }
            $message .= sprintf(
                __(' <a href="%s">Renew your license</a> to continue using premium features.', 'rlus'),
                esc_url('https://relayout.in/pricing')
            );
            ?>
            <div class="notice notice-error">
                <p><?php echo $message; ?></p>
            </div>
            <?php
        } else {
            // No license or invalid license
            ?>
            <div class="notice notice-warning">
                <p>
                    <?php 
                    printf(
                        __('%s requires a valid license key to function properly. <a href="%s">Enter your license key</a>', 'rlus'),
                        esc_html($product_name),
                        esc_url(admin_url('admin.php?page=rlus-settings'))
                    );
                    ?>
                </p>
            </div>
            <?php
        }
    }

    /**
     * Create license settings page
     *
     * @param string $page_title
     * @param string $menu_title
     * @param string $capability
     * @param string $menu_slug
     */
    public function create_license_page($page_title, $menu_title, $capability = 'manage_options', $menu_slug = '') {
        if (empty($menu_slug)) {
            $menu_slug = 'rlus-license-' . sanitize_title($this->product_id);
        }

        add_options_page(
            $page_title,
            $menu_title,
            $capability,
            $menu_slug,
            array($this, 'render_license_page')
        );
    }

    /**
     * Render license page
     */
    public function render_license_page() {
        $license_key = $this->get_stored_license_key();
        $license_info = $this->get_license_info();
        $is_valid = !empty($license_info);

        if (isset($_POST['submit'])) {
            check_admin_referer('rlus_sdk_license');
            
            $new_license_key = sanitize_text_field($_POST['license_key'] ?? '');
            
            if (!empty($new_license_key)) {
                $result = $this->activate_license($new_license_key);
                
                if ($result['success']) {
                    echo '<div class="notice notice-success"><p>' . __('License activated successfully!', 'rlus') . '</p></div>';
                    $license_key = $new_license_key;
                    $license_info = $this->get_license_info();
                    $is_valid = true;
                } else {
                    $error_message = '';
                    $check_result = $this->check_license($new_license_key);
                    
                    if (isset($check_result['is_expired']) && $check_result['is_expired'] === true) {
                        $error_message = __('This license has expired.', 'rlus');
                        if (isset($check_result['status_text']) && !empty($check_result['status_text'])) {
                            $error_message .= ' (' . esc_html($check_result['status_text']) . ')';
                        }
                        $error_message .= ' <a href="https://relayout.in/pricing" target="_blank">' . __('Renew your license', 'rlus') . '</a> ' . __('to continue using premium features.', 'rlus');
                    } elseif (isset($check_result['valid']) && $check_result['valid'] === false) {
                        $error_message = __('Invalid license key.', 'rlus');
                        if (isset($check_result['status_text']) && !empty($check_result['status_text'])) {
                            $error_message .= ' (' . esc_html($check_result['status_text']) . ')';
                        }
                        $error_message .= ' ' . __('Please check your license key or contact support.', 'rlus');
                    } else {
                        $error_message = $result['error'] ?? __('License activation failed', 'rlus');
                        if (empty(trim($error_message))) {
                            $error_message = __('License activation failed. Please check your connection and try again.', 'rlus');
                        }
                    }
                    
                    if (!empty($error_message)) {
                        echo '<div class="notice notice-error"><p>' . $error_message . '</p></div>';
                    } else {
                        echo '<div class="notice notice-error"><p>' . __('An unknown error occurred during license activation.', 'rlus') . '</p></div>';
                    }
                }
            } else {
                echo '<div class="notice notice-warning"><p>' . __('Please enter a license key.', 'rlus') . '</p></div>';
            }
        }

        if (isset($_POST['deactivate'])) {
            check_admin_referer('rlus_sdk_license');
            
            $result = $this->deactivate_license();
            
            if ($result['success']) {
                echo '<div class="notice notice-success"><p>' . __('License deactivated successfully!', 'rlus') . '</p></div>';
                $license_key = '';
                $license_info = array();
                $is_valid = false;
            } else {
                echo '<div class="notice notice-error"><p>' . esc_html($result['error']) . '</p></div>';
            }
        }

        ?>
        <style>
        .relayout-license-page {
            max-width: 1200px;
            margin: 20px 0;
        }
        .relayout-upgrade-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin: 30px 0;
        }
        .relayout-upgrade-card {
            background: white;
            border: 1px solid #e1e5e9;
            border-radius: 12px;
            padding: 30px;
            text-align: center;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        .relayout-upgrade-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        }
        .relayout-upgrade-card.featured {
            border: 2px solid #ff6b35;
            background: linear-gradient(135deg, #fff 0%, #fff8f6 100%);
        }
        .relayout-upgrade-card .plan-name {
            font-size: 24px;
            font-weight: 600;
            margin-bottom: 10px;
            color: #2c3e50;
        }
        .relayout-upgrade-card .plan-price {
            font-size: 36px;
            font-weight: 700;
            color: #ff6b35;
            margin-bottom: 20px;
        }
        .relayout-upgrade-card .plan-features {
            list-style: none;
            padding: 0;
            margin: 20px 0;
        }
        .relayout-upgrade-card .plan-features li {
            padding: 8px 0;
            border-bottom: 1px solid #f0f0f0;
        }
        .relayout-upgrade-card .plan-features li:before {
            content: "✓";
            color: #27ae60;
            font-weight: bold;
            margin-right: 10px;
        }
        .relayout-upgrade-btn {
            display: inline-block;
            background: #ff6b35;
            color: white;
            padding: 12px 30px;
            text-decoration: none;
            border-radius: 25px;
            font-weight: 600;
            transition: all 0.3s ease;
            border: none;
            cursor: pointer;
        }
        .relayout-upgrade-btn:hover {
            background: #e55a2b;
            transform: scale(1.05);
        }
        .relayout-license-form {
            background: white;
            padding: 30px;
            border-radius: 12px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-top: 30px;
        }
        </style>
        
        <div class="wrap relayout-license-page">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <?php if (!$is_valid): ?>
            <!-- Upgrade Cards Section -->
            <div class="relayout-upgrade-cards">
                <div class="relayout-upgrade-card">
                    <div class="plan-name">Personal</div>
                    <div class="plan-price">$29<small>/year</small></div>
                    <ul class="plan-features">
                        <li>All 9 Premium Widgets</li>
                        <li>1 Website License</li>
                        <li>Premium Email Support</li>
                        <li>Automatic Updates</li>
                    </ul>
                    <a href="https://relayout.in/pricing" class="relayout-upgrade-btn" target="_blank">Get Personal</a>
                </div>
                
                <div class="relayout-upgrade-card featured">
                    <div class="plan-name">Business</div>
                    <div class="plan-price">$59<small>/year</small></div>
                    <ul class="plan-features">
                        <li>All 9 Premium Widgets</li>
                        <li>5 Website License</li>
                        <li>Priority Support</li>
                        <li>Early Access Features</li>
                        <li>Priority Updates</li>
                    </ul>
                    <a href="https://relayout.in/pricing" class="relayout-upgrade-btn" target="_blank">Get Business</a>
                </div>
                
                <div class="relayout-upgrade-card">
                    <div class="plan-name">Agency</div>
                    <div class="plan-price">$99<small>/year</small></div>
                    <ul class="plan-features">
                        <li>All Premium Widgets</li>
                        <li>Unlimited Websites</li>
                        <li>White Label Option</li>
                        <li>48h Priority Support</li>
                        <li>Developer Resources</li>
                    </ul>
                    <a href="https://relayout.in/pricing" class="relayout-upgrade-btn" target="_blank">Get Agency</a>
                </div>
                
                <div class="relayout-upgrade-card" style="border: 2px solid #27ae60; background: linear-gradient(135deg, #fff 0%, #f0fff4 100%);">
                    <div class="plan-name" style="color: #27ae60;">Lifetime</div>
                    <div class="plan-price" style="color: #27ae60;">$149<small style="font-size: 14px;"> once</small></div>
                    <ul class="plan-features">
                        <li>All Premium Widgets</li>
                        <li>Unlimited Websites</li>
                        <li>Lifetime Updates</li>
                        <li>Lifetime Support</li>
                        <li>Early Adopter Bonus</li>
                    </ul>
                    <a href="https://relayout.in/pricing" class="relayout-upgrade-btn" target="_blank" style="background: #27ae60;">Get Lifetime</a>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- License Form -->
            <div class="relayout-license-form">
                <h2><?php _e('License Activation', 'rlus'); ?></h2>
                <form method="post" action="">
                    <?php wp_nonce_field('rlus_sdk_license'); ?>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php _e('License Key', 'rlus'); ?></th>
                            <td>
                                <input type="text" name="license_key" value="<?php echo esc_attr($license_key); ?>" class="regular-text" placeholder="Enter your license key here..." />
                                <p class="description"><?php _e('Enter your license key to activate this product.', 'rlus'); ?></p>
                            </td>
                        </tr>
                        
                        <?php if ($is_valid): ?>
                        <?php 
                        $license_status = $this->get_license_status();
                        $expiration_date = $this->get_expiration_date();
                        $status_text = $this->get_status_text();
                        ?>
                        <tr>
                            <th scope="row"><?php _e('License Status', 'rlus'); ?></th>
                            <td>
                                <?php if ($license_status['is_expired']): ?>
                                    <span style="color: #dc3232; font-weight: bold; background: #ffeaea; padding: 5px 10px; border-radius: 15px;"><?php _e('❌ Expired', 'rlus'); ?></span>
                                <?php elseif ($license_status['needs_renewal']): ?>
                                    <span style="color: #ff6b35; font-weight: bold; background: #fff4e6; padding: 5px 10px; border-radius: 15px;"><?php _e('⚠️ Expires Soon', 'rlus'); ?></span>
                                <?php else: ?>
                                    <span style="color: #27ae60; font-weight: bold; background: #d5f4e6; padding: 5px 10px; border-radius: 15px;"><?php _e('✅ Active', 'rlus'); ?></span>
                                <?php endif; ?>
                                
                                <?php if ($status_text): ?>
                                    <br><small style="margin-top: 10px; display: inline-block; color: #666;"><strong>Details:</strong> <?php echo esc_html($status_text); ?></small>
                                <?php endif; ?>
                                
                                <?php if (!empty($license_info['license_type'])): ?>
                                    <br><small style="margin-top: 10px; display: inline-block;"><strong>Type:</strong> <?php echo esc_html(ucwords(str_replace('-', ' ', $license_info['license_type']))); ?></small>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php if ($expiration_date): ?>
                        <tr>
                            <th scope="row"><?php _e('Expires', 'rlus'); ?></th>
                            <td>
                                <strong><?php echo esc_html($expiration_date); ?></strong>
                                <?php if ($license_status['days_remaining'] !== false): ?>
                                    <?php if ($license_status['is_expired']): ?>
                                        <span style="color: #dc3232; font-weight: bold; margin-left: 10px;">(Expired)</span>
                                    <?php elseif ($license_status['days_remaining'] <= 1): ?>
                                        <span style="color: #ff6b35; font-weight: bold; margin-left: 10px;">(<?php echo $license_status['days_remaining']; ?> day<?php echo $license_status['days_remaining'] != 1 ? 's' : ''; ?> remaining)</span>
                                        <br><small style="color: #ff6b35;">⚠️ Renewal recommended soon</small>
                                    <?php else: ?>
                                        <span style="color: #27ae60; margin-left: 10px;">(<?php echo $license_status['days_remaining']; ?> day<?php echo $license_status['days_remaining'] != 1 ? 's' : ''; ?> remaining)</span>
                                    <?php endif; ?>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endif; ?>
                        <?php endif; ?>
                    </table>
                    
                    <p class="submit">
                        <?php if (!$is_valid): ?>
                            <button type="submit" name="submit" class="button button-primary button-large">
                                <?php _e('Activate License', 'rlus'); ?>
                            </button>
                        <?php else: ?>
                            <button type="submit" name="submit" class="button button-primary">
                                <?php _e('Update License', 'rlus'); ?>
                            </button>
                            <button type="submit" name="deactivate" class="button button-secondary" onclick="return confirm('<?php _e('Are you sure you want to deactivate this license?', 'rlus'); ?>')">
                                <?php _e('Deactivate License', 'rlus'); ?>
                            </button>
                        <?php endif; ?>
                    </p>
                </form>
            </div>
        </div>
        <?php
    }
}
